/* dd -- convert a file while copying it.
   Copyright (C) 85, 90, 91, 1995-2008 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

/* Written by Paul Rubin, David MacKenzie, and Stuart Kemp. */

#include <config.h>

#define SWAB_ALIGN_OFFSET 2

#include <sys/types.h>
#include <signal.h>
#include <getopt.h>

#include "system.h"
#include "error.h"
#include "fd-reopen.h"
#include "gethrxtime.h"
#include "human.h"
#include "long-options.h"
#include "quote.h"
#include "quotearg.h"
#include "xstrtol.h"
#include "xtime.h"

#include "md5.h"
#include "sha1.h"
#include "sha256.h"
#include "sha512.h"

#include <stdarg.h>

static void process_signals (void);

/* The official name of this program (e.g., no `g' prefix).  */
#define PROGRAM_NAME "dc3dd"

#define AUTHORS \
  proper_name ("Paul Rubin"), \
  proper_name ("David MacKenzie"), \
  proper_name ("Stuart Kemp"), \
  "patched for DC3 by Jesse Kornblum and Andrew Medico"

/* Use SA_NOCLDSTOP as a proxy for whether the sigaction machinery is
   present.  SA_NODEFER and SA_RESETHAND are XSI extensions.  */
#ifndef SA_NOCLDSTOP
# define SA_NOCLDSTOP 0
# define sigprocmask(How, Set, Oset) /* empty */
# define sigset_t int
# if ! HAVE_SIGINTERRUPT
#  define siginterrupt(sig, flag) /* empty */
# endif
#endif
#ifndef SA_NODEFER
# define SA_NODEFER 0
#endif
#ifndef SA_RESETHAND
# define SA_RESETHAND 0
#endif

#ifndef SIGINFO
# define SIGINFO SIGUSR1
#endif

#if ! HAVE_FDATASYNC
# define fdatasync(fd) (errno = ENOSYS, -1)
#endif

#define max(a, b) ((a) > (b) ? (a) : (b))
#define output_char(c)				\
  do						\
    {						\
      obuf[oc++] = (c);				\
      if (oc >= output_blocksize)		\
	write_output ();			\
    }						\
  while (0)

/* Default input and output blocksize. */
#ifndef DEFAULT_BLOCKSIZE
# define DEFAULT_BLOCKSIZE 32768
#endif

/* How many bytes to add to the input and output block sizes before invoking
   malloc.  See dd_copy for details.  INPUT_BLOCK_SLOP must be no less than
   OUTPUT_BLOCK_SLOP.  */
#define INPUT_BLOCK_SLOP (2 * SWAB_ALIGN_OFFSET + 2 * page_size - 1)
#define OUTPUT_BLOCK_SLOP (page_size - 1)

/* Maximum blocksize for the given SLOP.
   Keep it smaller than SIZE_MAX - SLOP, so that we can
   allocate buffers that size.  Keep it smaller than SSIZE_MAX, for
   the benefit of system calls like "read".  And keep it smaller than
   OFF_T_MAX, for the benefit of the large-offset seek code.  */
#define MAX_BLOCKSIZE(slop) MIN (SIZE_MAX - (slop), MIN (SSIZE_MAX, OFF_T_MAX))

/* Conversions bit masks. */
enum
  {
    C_ASCII = 01,

    C_EBCDIC = 02,
    C_IBM = 04,
    C_BLOCK = 010,
    C_UNBLOCK = 020,
    C_LCASE = 040,
    C_UCASE = 0100,
    C_SWAB = 0200,
    C_NOERROR = 0400,
    C_NOTRUNC = 01000,
    C_SYNC = 02000,

    /* Use separate input and output buffers, and combine partial
       input blocks. */
    C_TWOBUFS = 04000,

    C_NOCREAT = 010000,
    C_EXCL = 020000,
    C_FDATASYNC = 040000,
    C_FSYNC = 0100000,

    C_DYNAMIC = 0200000
  };

/* Status bit masks.  */
enum
  {
    STATUS_NOXFER = 01
  };

/* The name this program was run with. */
char *program_name;

// TODO: need to resolve the desire for input_file/output_file to be
// const, with the fact that split/join modify the values
/* The name of the input file, or NULL for the standard input. */
static char /*const*/ *input_file = NULL;

/* The name of the output file, or NULL for the standard output. */
static char /*const*/ *output_file = NULL;

/* The page size on this host.  */
static size_t page_size;

/* The number of bytes in which atomic reads are done. */
static size_t input_blocksize = 0;

/* The number of bytes in which atomic writes are done. */
static size_t output_blocksize = 0;

/* The sector size of the applicable device
 * (input device normally; output device when in pattern mode) */
static size_t dc3_sectorsize = 0;

static size_t dc3_small_read = 0;

/* Conversion buffer size, in bytes.  0 prevents conversions. */
static size_t conversion_blocksize = 0;

/* Skip this many records of `input_blocksize' bytes before input. */
static uintmax_t skip_sectors = 0;

/* Skip this many records of `output_blocksize' bytes before output. */
static uintmax_t seek_sectors = 0;

/* Copy only this many records.  The default is effectively infinity.  */
static uintmax_t max_sectors = (uintmax_t) -1;

/* Bit vector of conversions to apply. */
static int conversions_mask = 0;

/* Open flags for the input and output files.  */
static int input_flags = 0;
static int output_flags = 0;

/* Status flags for what is printed to stderr.  */
static int status_flags = 0;

/* If nonzero, filter characters through the translation table.  */
static bool translation_needed = false;

/* Number of partial blocks read. */
static uintmax_t r_partial = 0;

/* Number of full blocks read. */
static uintmax_t r_full = 0;

/* Number of bytes written.  */
static uintmax_t w_bytes = 0;

/* Number of bytes in input file */
static uintmax_t input_bytes = 0;

static uintmax_t dc3_sectors_w_partial = 0;
static uintmax_t dc3_sectors_w_full    = 0;
static uintmax_t dc3_sectors_r_partial = 0;
static uintmax_t dc3_sectors_r_full    = 0;

/* Whether or not we should probe the input file and get its size */
#ifdef DEFAULT_SIZEPROBE
  static bool dc3_sizeprobe = true;
#else
  static bool dc3_sizeprobe = false;
#endif

/* Time that dd started.  */
static xtime_t start_time;

/* True if input is seekable.  */
static bool input_seekable;

/* True if we're wiping a device */
static bool dc3_mode_wipe = false;

/* True if we're verifying the output file */
static bool dc3_mode_verify = false;


/* True if we should be combining error messages */
static bool dc3_group_errors = false;

/* True if the last read was an error */
static bool dc3_previous_error = false;

/* Number of previous reads that were errors */
static uintmax_t dc3_previous_error_count = 0;

/* Previous error encountered */
static int dc3_previous_error_errno = 0;


/* True if we're piping the output to another process */
static bool dc3_pipe_output = false;

/* True if we are splitting the output file */
static bool dc3_split = false;

/* Number of bytes that should be written to each split file */
uintmax_t dc3_split_bytes = 0;

/* Number of bytes written to the current split file */
uintmax_t dc3_split_current = 0;

/* Format for split files */
char * dc3_split_format = "000";

/* Current number of split files */
uintmax_t dc3_split_number = 0;

/* Current filename being written to */
char * dc3_split_filename = NULL;

// ifjoin state
static bool dc3_ifjoin        = false; // Are we reading in split files?
char*       dc3_ifjoin_base   = NULL;  // Base name for reading in split files
char*       dc3_ifjoin_ext    = NULL;  // splitformat for reading in split files
uintmax_t   dc3_ifjoin_number = 0;     // Current index of input split file
static bool dc3_ifjoin_done   = false; // True if we closed the last input split file

// vfjoin state
static bool dc3_vfjoin        = false; // Are we verifying split files?
char*       dc3_vfjoin_base   = NULL;  // Base name for verifying split files
char*       dc3_vfjoin_ext    = NULL;  // splitformat for verifying split files
uintmax_t   dc3_vfjoin_number = 0;     // Current index of verify split file

/* True if we need to compute hashes */
static bool dc3_hash = false;

/* True if the user has already specified a hash algorithm to use */
static bool dc3_hash_manual = false;

/* Bytes to use for each piecewise hash */
uintmax_t dc3_hash_window = 0;

/* Bytes read in the current piecewise hash */
uintmax_t dc3_hash_current = 0;

/* True if we need to compute hashes before copy (hashconv=before) */
#ifdef DEFAULT_HASHCONV_BEFORE
  static bool dc3_hash_before = true;
#else 
# ifdef DEFAULT_HASHCONV_AFTER
   static bool dc3_hash_before = false;
#  else
   static bool dc3_hash_before = true;
# endif
#endif

/* Holds data used to update each hash */
static char * dc3_hash_buffer;


/* External log for hashes */
FILE * hash_log = NULL;

/* External log for errors */
FILE * error_log = NULL;


/* True if we need to display status messages */
#ifdef DEFAULT_PROGRESS
 static bool dc3_progress = true;
#else
 static bool dc3_progress = false;
#endif

/* Becomes true once a progress message has been printed */
static bool dc3_progress_printed = false;

/* Number of blocks counted before a progress count is displayed */
#ifdef DEFAULT_PROGRESSCOUNT
  static uintmax_t dc3_progress_count   = (uintmax_t)DEFAULT_PROGRESSCOUNT;
#else
  static uintmax_t dc3_progress_count   = 0;
#endif

/* Number of blocks read since the last progress message was displayed */
static uintmax_t dc3_progress_current = 0;


/* True if we're using a pattern as input */
static bool dc3_use_pattern = false;

enum
  {
    DC3_EXIT_COMPLETE = 01,
    DC3_EXIT_ABORTED  = 02,
    DC3_EXIT_FAILED   = 04
  };

static int dc3_exit_reason = 0;


/* We use the following datatype to keep track of the state of
   the various hashing algorithms. 

   To add a new algorithm:

   1. Increment DC3_NUM_HASHES

   2. Add the appropriate definition to dc3_hashinfo.
      This includes the size of the hash sum (i.e. the size 
      of the hash in bytes, bit length divided by eight),
      the size of the internal hash state structure,
      and the functions necessary to initialize, update,
      and finalize the hash. 
*/


/* State for hashing algorihtms */
typedef struct _dc3_hashtype_t {
  bool    inuse;

  /* The name of the hash algorithm, e.g. "md5" */
  char  * name;

  /* We have to keep separate contexts for the current window of
     data being hashed and the overall file. These blocks start
     at different points. */
  void  * ctx_window;
  void  * ctx_total;

  /* We can get away with just one sum and result buffer, however,
     as these get computed at separate times for the current window
     and the file as a whole. */
  char  * sum;
  char  * result;

  /* The sum_size is used to compute the size of the result. 
     Size of Result = 2 * sum_size */
  size_t  sum_size;
  
  /* Size of the internal data structures stored in ctx_window and
     and ctx_total. This is used to allocate memory. */
  size_t  ctx_size;

  /* Number of bytes processed in total. Used to compute the 
     start and stop points for each hash window. */
  uintmax_t total;
  
  void (*init)(void *);
  void (*update)(const void *, size_t, void *);
  void (*final)(void *, void *);

} dc3_hashtype_t;


#define DC3_NUM_HASHES   4

static dc3_hashtype_t dc3_hashinfo[DC3_NUM_HASHES] = 
  {
    { 
#ifdef DEFAULT_HASH_MD5
      true,
#else
      false,
#endif
      "md5",
      NULL,
      NULL,
      NULL,
      NULL,
      16,
      sizeof(struct md5_ctx),
      0,
      (void (*)(void *))md5_init_ctx,
      (void (*)(const void *, size_t, void *))md5_process_bytes,
      (void (*)(void *, void *))md5_finish_ctx},
    
    {
#ifdef DEFAULT_HASH_SHA1
      true,
#else
      false,
#endif
      "sha1",
      NULL,
      NULL,
      NULL,
      NULL,
      20,
      sizeof(struct sha1_ctx),
      0,
      (void (*)(void *))sha1_init_ctx,
      (void (*)(const void *, size_t, void *))sha1_process_bytes,
      (void (*)(void *, void *))sha1_finish_ctx},
    
    {
#ifdef DEFAULT_HASH_SHA256
      true,
#else
      false,
#endif
      "sha256",
      NULL,
      NULL,
      NULL,
      NULL,
      32,
      sizeof(struct sha256_ctx),
      0,
      (void (*)(void *))sha256_init_ctx,
      (void (*)(const void *, size_t, void *))sha256_process_bytes,
      (void (*)(void *, void *))sha256_finish_ctx},
    
    {
#ifdef DEFAULT_HASH_SHA512
      true,
#else
      false,
#endif
     "sha512",
      NULL,
      NULL,
      NULL,
      NULL,
      64,
      sizeof(struct sha512_ctx),
      0,
      (void (*)(void *))sha512_init_ctx,
      (void (*)(const void *, size_t, void *))sha512_process_bytes,
      (void (*)(void *, void *))sha512_finish_ctx}
  };


static void 
dc3_error(int status, int errnum, const char *message, ...);

void dc3_exit_msg();

static void dc3_parse_hash(const char *val)
{
  bool done = false;
  uint8_t i;
  char *new;

  /* If there have been algorithms specified using the configure 
     script (e.g. -DDEFAULT_HASH_MD5), we want to clear those
     *unless* this is the second time through this function. 
     (e.g. the user called hash=md5 hash=sha1). */
  if (!dc3_hash_manual)
    {
      for (i = 0 ; i < DC3_NUM_HASHES ; i++)
	dc3_hashinfo[i].inuse = false;
    }
  dc3_hash_manual = true;
  
  do {
    new = strchr(val,',');
    if (NULL == new)
      done = true;
    else
      *new = 0;
    
    for (i = 0 ; i < DC3_NUM_HASHES ; i++)
      {
	if (STREQ(dc3_hashinfo[i].name,val))
	  {
	    dc3_hashinfo[i].inuse = true;
	    break;
	  }
      }
    if (DC3_NUM_HASHES == i)
      dc3_error(EXIT_FAILURE,0,_("Unknown hash algorithm %s"), quote(val));
    
    val = new+1;
  } while (!done);
}

static void
dc3_hash_init_algorithm(uint8_t i)
{
  dc3_hashinfo[i].ctx_window = malloc(dc3_hashinfo[i].ctx_size);
  dc3_hashinfo[i].ctx_total  = malloc(dc3_hashinfo[i].ctx_size);
  dc3_hashinfo[i].sum        = malloc(dc3_hashinfo[i].sum_size);
  dc3_hashinfo[i].result     = malloc(2 * dc3_hashinfo[i].sum_size + 1);

  if (NULL == dc3_hashinfo[i].ctx_window ||
      NULL == dc3_hashinfo[i].ctx_total  ||
      NULL == dc3_hashinfo[i].sum        ||
      NULL == dc3_hashinfo[i].result)
    dc3_error(EXIT_FAILURE,0,_("Unable to allocate space for hashes"));

  dc3_hash_current = 0;

  dc3_hashinfo[i].init(dc3_hashinfo[i].ctx_window);
  dc3_hashinfo[i].init(dc3_hashinfo[i].ctx_total);
}


static void
dc3_hash_init(void)
{
  uint8_t i;

  if (NULL == hash_log)
    hash_log = stderr;

  for (i = 0 ; i < DC3_NUM_HASHES ; ++i)
    {
      if (dc3_hashinfo[i].inuse)
	dc3_hash_init_algorithm(i);
    }
}



static void
dc3_hash_update_total(uint8_t i, char *buf, size_t len)
{
  memcpy(dc3_hash_buffer,buf,len);
  dc3_hashinfo[i].update(dc3_hash_buffer,len,dc3_hashinfo[i].ctx_total);
}


static void
dc3_hash_result(uint8_t i)
{
  size_t p;
  static char hex[] = "0123456789abcdef";

  for (p = 0; p < dc3_hashinfo[i].sum_size ; p++)
    {
      dc3_hashinfo[i].result[2 * p] = hex[(dc3_hashinfo[i].sum[p] >> 4) & 0xf];
      dc3_hashinfo[i].result[2 * p + 1] = hex[dc3_hashinfo[i].sum[p] & 0xf];
    }

  dc3_hashinfo[i].result[2 * dc3_hashinfo[i].sum_size] = 0;
}

static void
dc3_hash_display_window_print(uint8_t i, FILE* file)
{
  fprintf(file,"%s %"PRIuMAX"- %"PRIuMAX": %s\n",
	  dc3_hashinfo[i].name,
	  dc3_hashinfo[i].total - dc3_hash_current,
	  dc3_hashinfo[i].total,
	  dc3_hashinfo[i].result);
  fflush(file);
}

static void
dc3_hash_display_window(uint8_t i)
{ 
  dc3_hash_result(i);

  // TODO: really, this only needs to be done before the *first* hash line, but it shouldn't hurt to do it for each one
  fprintf(stderr, "                                                                   \r");
  dc3_hash_display_window_print(i, stderr);
  if (hash_log != stderr)
    dc3_hash_display_window_print(i, hash_log);
}

static void
dc3_hash_display_final_print(uint8_t i, FILE* file)
{
  fprintf(file,"%s TOTAL: %s\n",  
	  dc3_hashinfo[i].name,
	  dc3_hashinfo[i].result);
  fflush(file);
}

static void
dc3_hash_display_final(uint8_t i)
{
  dc3_hash_result(i);

  dc3_hash_display_final_print(i, stderr);
  if (hash_log != stderr)
    dc3_hash_display_final_print(i, hash_log);
}

static void
dc3_hash_update_window(char *buf, size_t len)
{
  uint8_t i;
  uintmax_t left = dc3_hash_window - dc3_hash_current;

  //fprintf(stderr, "dc3_hash_display_window() called with len=%zd, dc3_hash_current=%zd\n", len, dc3_hash_current);

  if (0 == left)
    {
      for (i = 0 ; i < DC3_NUM_HASHES ; ++i)
	{
	  if (dc3_hashinfo[i].inuse)
	    {
	      dc3_hashinfo[i].final(dc3_hashinfo[i].ctx_window,
				    dc3_hashinfo[i].sum);
	      dc3_hash_display_window(i);
	      dc3_hashinfo[i].init(dc3_hashinfo[i].ctx_window);
	    }
	}

      //fprintf(stderr, "dc3_hash_update_window() resetting dc3_hash_current to 0\n");

      dc3_hash_current = 0;
      left = dc3_hash_window;
    }

  memcpy(dc3_hash_buffer,buf,len);

  if (len <= left)
    {
      for (i = 0 ; i < DC3_NUM_HASHES ; ++i)
	{
	  if (dc3_hashinfo[i].inuse)
	    {
	      dc3_hashinfo[i].update(dc3_hash_buffer,
				     len,
				     dc3_hashinfo[i].ctx_window);

	      // RBF - Should total calculations to be moved total_update function?
	      dc3_hashinfo[i].total   += len;
	    }
	}

      //fprintf(stderr, "dc3_hash_update_window() 1 incrementing dc3_hash_current by len %zd\n", len);
      dc3_hash_current += len;

    }
  else
    {
      for (i = 0 ; i < DC3_NUM_HASHES ; ++i)
	{
	  if (dc3_hashinfo[i].inuse)
	    {
	      dc3_hashinfo[i].update(dc3_hash_buffer,
				     left,
				     dc3_hashinfo[i].ctx_window);
	      // RBF - Should total calculations to be moved total_update function?
	      dc3_hashinfo[i].total   += left;
	    }
	}

      //fprintf(stderr, "dc3_hash_update_window() 2 incrementing dc3_hash_current by left %zd\n", left);
      dc3_hash_current += left;
      dc3_hash_update_window(buf+left,len-left);
    }

  //fprintf(stderr, "dc3_hash_update_window() final: dc3_hash_current now %zd\n", dc3_hash_current);
}


static void
dc3_hash_update(char *buf, size_t len)
{
  uint8_t i;
  
  dc3_hash_buffer = malloc(len);
  if (NULL == dc3_hash_buffer)
    error(EXIT_FAILURE,0,_("Unable to allocate hashing buffer"));

  //fprintf(stderr, "dc3_hash_update: called with len=%zd\n", len);

  /* It's easy to update the total hash, so we do that first. */

  for (i = 0 ; i < DC3_NUM_HASHES ; ++i)
    {
      if (dc3_hashinfo[i].inuse)
	dc3_hash_update_total(i,buf,len);
    }  

  if (dc3_hash_window != 0)
    dc3_hash_update_window(buf,len);

  free(dc3_hash_buffer);
}

static void
dc3_hash_final(void)
{
  //fprintf(stderr, "dc3_hash_final called\n");
  uint8_t i;

  fprintf(stderr, "                                                                   \r");

  /* We want to display all of the piecewise hashes before we
     display any of the total hashes */
  if (dc3_hash_window != 0)
    {
      for (i = 0 ; i < DC3_NUM_HASHES ; ++i)
	{
	  if (dc3_hashinfo[i].inuse)
	    {
	      dc3_hashinfo[i].final(dc3_hashinfo[i].ctx_window,
				    dc3_hashinfo[i].sum);
	      dc3_hash_display_window(i);
	    }
	}
    }  
  
  for (i = 0 ; i < DC3_NUM_HASHES ; ++i)
    {
      if (dc3_hashinfo[i].inuse)
	{
          //fprintf(stderr, "dc3_hash_final(): calling final() for hash %d, ctx_total=%zd\n", i, dc3_hashinfo[i].ctx_total);
	  dc3_hashinfo[i].final(dc3_hashinfo[i].ctx_total,
				dc3_hashinfo[i].sum);
	  dc3_hash_display_final(i);
	}
    }  
}

/* Get current time as a string
 * Format: 'YYYY-MM-DD HH:MM:SS -0000'
 */
char*
dc3_time()
{
  const size_t len = 32; // more than enough to hold 'YYYY-MM-DD HH:MM:SS -0000'
  time_t t;
  struct tm tm;
  char* time_str = xmalloc(len);

  t = time(NULL);
  struct tm* ret = localtime_r(&t, &tm);

  if (ret == NULL)
    dc3_error(EXIT_FAILURE, errno, "localtime");

  if (strftime(time_str, len, "%F %T %z", &tm) == 0)
    dc3_error(EXIT_FAILURE, 0, "strftime returned 0");

  return time_str;
}

// return the extension of the given filename, after splitting
// on the given delimiter
const char*
dc3_fileext(const char* filename, const char* delim)
{
    const char* next = filename;
    const char* ext = NULL;

    while ((next = strstr(next, delim)) != NULL)
    {
        next = ext = next + 1;
    }

    return ext;
}

const char*
dc3_filebase(const char* filename, const char* delim)
{
    const char* next = filename;
    const char* ext = NULL;

    while ((next = strstr(next, delim)) != NULL)
    {
        next = ext = next + 1;
    }

    if (ext == NULL)
	return NULL;

    int offset = (ext-1) - filename;

    char* base = xstrdup(filename);
    base[offset] = '\0';

    return base;
}

bool dc3_validate_splitformat(const char* pattern)
{
    size_t pos, len = strlen(pattern);
    char first      = tolower(pattern[0]);

    for (pos = 0; pos < len; ++pos)
    {
        if ((tolower(pattern[pos]) != 'a' &&
                    tolower(pattern[pos]) != '1' &&
                    tolower(pattern[pos]) != '0') ||
                tolower(pattern[pos]) != first)
            return false;
    }

    return true;
}

/* Pattern used as input */
char * dc3_pattern_value;

/* Length of the pattern being used */
uintmax_t dc3_pattern_len;

/* Error number corresponding to initial attempt to lseek input.
   If ESPIPE, do not issue any more diagnostics about it.  */
static int input_seek_errno;

/* File offset of the input, in bytes, along with a flag recording
   whether it overflowed.  The offset is valid only if the input is
   seekable and if the offset has not overflowed.  */
static uintmax_t input_offset;
static bool input_offset_overflow;

/* Records truncated by conv=block. */
static uintmax_t r_truncate = 0;

/* Output representation of newline and space characters.
   They change if we're converting to EBCDIC.  */
static char newline_character = '\n';
static char space_character = ' ';

/* Output buffer. */
static char *obuf;

/* Current index into `obuf'. */
static size_t oc = 0;

/* Index into current line, for `conv=block' and `conv=unblock'.  */
static size_t col = 0;

/* The set of signals that are caught.  */
static sigset_t caught_signals;

/* If nonzero, the value of the pending fatal signal.  */
static sig_atomic_t volatile interrupt_signal;

/* A count of the number of pending info signals that have been received.  */
static sig_atomic_t volatile info_signal_count;

/* A longest symbol in the struct symbol_values tables below.  */
#define LONGEST_SYMBOL "fdatasync"

/* A symbol and the corresponding integer value.  */
struct symbol_value
{
  char symbol[sizeof LONGEST_SYMBOL];
  int value;
};

/* Conversion symbols, for conv="...".  */
static struct symbol_value const conversions[] =
{
  {"ascii", C_ASCII | C_TWOBUFS},	/* EBCDIC to ASCII. */
  {"ebcdic", C_EBCDIC | C_TWOBUFS},	/* ASCII to EBCDIC. */
  {"ibm", C_IBM | C_TWOBUFS},	/* Slightly different ASCII to EBCDIC. */
  {"block", C_BLOCK | C_TWOBUFS},	/* Variable to fixed length records. */
  {"unblock", C_UNBLOCK | C_TWOBUFS},	/* Fixed to variable length records. */
  {"lcase", C_LCASE | C_TWOBUFS},	/* Translate upper to lower case. */
  {"ucase", C_UCASE | C_TWOBUFS},	/* Translate lower to upper case. */
  {"swab", C_SWAB | C_TWOBUFS},	/* Swap bytes of input. */
  {"noerror", C_NOERROR},	/* Ignore i/o errors. */
  {"nocreat", C_NOCREAT},	/* Do not create output file.  */
  {"excl", C_EXCL},		/* Fail if the output file already exists.  */
  {"notrunc", C_NOTRUNC},	/* Do not truncate output file. */
  {"sync", C_SYNC},		/* Pad input records to ibs with NULs. */
  {"fdatasync", C_FDATASYNC},	/* Synchronize output data before finishing.  */
  {"fsync", C_FSYNC},		/* Also synchronize output metadata.  */
  //{"dynamic", C_DYNAMIC},       /* When read errors occur on input, try to
  //                                 re-read smaller pieces of the block to
  //                                 maximize data recovery */
  {"", 0}
};

/* Flags, for iflag="..." and oflag="...".  */
static struct symbol_value const flags[] =
{
  {"append",	O_APPEND},
  {"binary",	O_BINARY},
  {"direct",	O_DIRECT},
  {"directory",	O_DIRECTORY},
  {"dsync",	O_DSYNC},
  {"noatime",	O_NOATIME},
  {"noctty",	O_NOCTTY},
  {"nofollow",	HAVE_WORKING_O_NOFOLLOW ? O_NOFOLLOW : 0},
  {"nolinks",	O_NOLINKS},
  {"nonblock",	O_NONBLOCK},
  {"sync",	O_SYNC},
  {"text",	O_TEXT},
  {"",		0}
};

/* Status, for status="...".  */
static struct symbol_value const statuses[] =
{
  {"noxfer",	STATUS_NOXFER},
  {"",		0}
};

/* Translation table formed by applying successive transformations. */
static unsigned char trans_table[256];

static char const ascii_to_ebcdic[] =
{
  '\000', '\001', '\002', '\003', '\067', '\055', '\056', '\057',
  '\026', '\005', '\045', '\013', '\014', '\015', '\016', '\017',
  '\020', '\021', '\022', '\023', '\074', '\075', '\062', '\046',
  '\030', '\031', '\077', '\047', '\034', '\035', '\036', '\037',
  '\100', '\117', '\177', '\173', '\133', '\154', '\120', '\175',
  '\115', '\135', '\134', '\116', '\153', '\140', '\113', '\141',
  '\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
  '\370', '\371', '\172', '\136', '\114', '\176', '\156', '\157',
  '\174', '\301', '\302', '\303', '\304', '\305', '\306', '\307',
  '\310', '\311', '\321', '\322', '\323', '\324', '\325', '\326',
  '\327', '\330', '\331', '\342', '\343', '\344', '\345', '\346',
  '\347', '\350', '\351', '\112', '\340', '\132', '\137', '\155',
  '\171', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
  '\210', '\211', '\221', '\222', '\223', '\224', '\225', '\226',
  '\227', '\230', '\231', '\242', '\243', '\244', '\245', '\246',
  '\247', '\250', '\251', '\300', '\152', '\320', '\241', '\007',
  '\040', '\041', '\042', '\043', '\044', '\025', '\006', '\027',
  '\050', '\051', '\052', '\053', '\054', '\011', '\012', '\033',
  '\060', '\061', '\032', '\063', '\064', '\065', '\066', '\010',
  '\070', '\071', '\072', '\073', '\004', '\024', '\076', '\341',
  '\101', '\102', '\103', '\104', '\105', '\106', '\107', '\110',
  '\111', '\121', '\122', '\123', '\124', '\125', '\126', '\127',
  '\130', '\131', '\142', '\143', '\144', '\145', '\146', '\147',
  '\150', '\151', '\160', '\161', '\162', '\163', '\164', '\165',
  '\166', '\167', '\170', '\200', '\212', '\213', '\214', '\215',
  '\216', '\217', '\220', '\232', '\233', '\234', '\235', '\236',
  '\237', '\240', '\252', '\253', '\254', '\255', '\256', '\257',
  '\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
  '\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
  '\312', '\313', '\314', '\315', '\316', '\317', '\332', '\333',
  '\334', '\335', '\336', '\337', '\352', '\353', '\354', '\355',
  '\356', '\357', '\372', '\373', '\374', '\375', '\376', '\377'
};

static char const ascii_to_ibm[] =
{
  '\000', '\001', '\002', '\003', '\067', '\055', '\056', '\057',
  '\026', '\005', '\045', '\013', '\014', '\015', '\016', '\017',
  '\020', '\021', '\022', '\023', '\074', '\075', '\062', '\046',
  '\030', '\031', '\077', '\047', '\034', '\035', '\036', '\037',
  '\100', '\132', '\177', '\173', '\133', '\154', '\120', '\175',
  '\115', '\135', '\134', '\116', '\153', '\140', '\113', '\141',
  '\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
  '\370', '\371', '\172', '\136', '\114', '\176', '\156', '\157',
  '\174', '\301', '\302', '\303', '\304', '\305', '\306', '\307',
  '\310', '\311', '\321', '\322', '\323', '\324', '\325', '\326',
  '\327', '\330', '\331', '\342', '\343', '\344', '\345', '\346',
  '\347', '\350', '\351', '\255', '\340', '\275', '\137', '\155',
  '\171', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
  '\210', '\211', '\221', '\222', '\223', '\224', '\225', '\226',
  '\227', '\230', '\231', '\242', '\243', '\244', '\245', '\246',
  '\247', '\250', '\251', '\300', '\117', '\320', '\241', '\007',
  '\040', '\041', '\042', '\043', '\044', '\025', '\006', '\027',
  '\050', '\051', '\052', '\053', '\054', '\011', '\012', '\033',
  '\060', '\061', '\032', '\063', '\064', '\065', '\066', '\010',
  '\070', '\071', '\072', '\073', '\004', '\024', '\076', '\341',
  '\101', '\102', '\103', '\104', '\105', '\106', '\107', '\110',
  '\111', '\121', '\122', '\123', '\124', '\125', '\126', '\127',
  '\130', '\131', '\142', '\143', '\144', '\145', '\146', '\147',
  '\150', '\151', '\160', '\161', '\162', '\163', '\164', '\165',
  '\166', '\167', '\170', '\200', '\212', '\213', '\214', '\215',
  '\216', '\217', '\220', '\232', '\233', '\234', '\235', '\236',
  '\237', '\240', '\252', '\253', '\254', '\255', '\256', '\257',
  '\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
  '\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
  '\312', '\313', '\314', '\315', '\316', '\317', '\332', '\333',
  '\334', '\335', '\336', '\337', '\352', '\353', '\354', '\355',
  '\356', '\357', '\372', '\373', '\374', '\375', '\376', '\377'
};

static char const ebcdic_to_ascii[] =
{
  '\000', '\001', '\002', '\003', '\234', '\011', '\206', '\177',
  '\227', '\215', '\216', '\013', '\014', '\015', '\016', '\017',
  '\020', '\021', '\022', '\023', '\235', '\205', '\010', '\207',
  '\030', '\031', '\222', '\217', '\034', '\035', '\036', '\037',
  '\200', '\201', '\202', '\203', '\204', '\012', '\027', '\033',
  '\210', '\211', '\212', '\213', '\214', '\005', '\006', '\007',
  '\220', '\221', '\026', '\223', '\224', '\225', '\226', '\004',
  '\230', '\231', '\232', '\233', '\024', '\025', '\236', '\032',
  '\040', '\240', '\241', '\242', '\243', '\244', '\245', '\246',
  '\247', '\250', '\133', '\056', '\074', '\050', '\053', '\041',
  '\046', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
  '\260', '\261', '\135', '\044', '\052', '\051', '\073', '\136',
  '\055', '\057', '\262', '\263', '\264', '\265', '\266', '\267',
  '\270', '\271', '\174', '\054', '\045', '\137', '\076', '\077',
  '\272', '\273', '\274', '\275', '\276', '\277', '\300', '\301',
  '\302', '\140', '\072', '\043', '\100', '\047', '\075', '\042',
  '\303', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
  '\150', '\151', '\304', '\305', '\306', '\307', '\310', '\311',
  '\312', '\152', '\153', '\154', '\155', '\156', '\157', '\160',
  '\161', '\162', '\313', '\314', '\315', '\316', '\317', '\320',
  '\321', '\176', '\163', '\164', '\165', '\166', '\167', '\170',
  '\171', '\172', '\322', '\323', '\324', '\325', '\326', '\327',
  '\330', '\331', '\332', '\333', '\334', '\335', '\336', '\337',
  '\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
  '\173', '\101', '\102', '\103', '\104', '\105', '\106', '\107',
  '\110', '\111', '\350', '\351', '\352', '\353', '\354', '\355',
  '\175', '\112', '\113', '\114', '\115', '\116', '\117', '\120',
  '\121', '\122', '\356', '\357', '\360', '\361', '\362', '\363',
  '\134', '\237', '\123', '\124', '\125', '\126', '\127', '\130',
  '\131', '\132', '\364', '\365', '\366', '\367', '\370', '\371',
  '\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
  '\070', '\071', '\372', '\373', '\374', '\375', '\376', '\377'
};

/* True if we need to close the standard output *stream*.  */
static bool close_stdout_required = true;

/* The only reason to close the standard output *stream* is if
   parse_long_options fails (as it does for --help or --version).
   In any other case, dd uses only the STDOUT_FILENO file descriptor,
   and the "cleanup" function calls "close (STDOUT_FILENO)".
   Closing the file descriptor and then letting the usual atexit-run
   close_stdout function call "fclose (stdout)" would result in a
   harmless failure of the close syscall (with errno EBADF).
   This function serves solely to avoid the unnecessary close_stdout
   call, once parse_long_options has succeeded.  */
static void
maybe_close_stdout (void)
{
  if (close_stdout_required)
    close_stdout ();
}

void
usage (int status)
{
  if (status != EXIT_SUCCESS)
    fprintf (stderr, _("Try `%s --help' for more information.\n"),
	     program_name);
  else
    {
      printf (_("\
Usage: %s [OPERAND]...\n\
  or:  %s OPTION\n\
"),
	      program_name, program_name);
      fputs (_("\
Copy a file, converting and formatting according to the operands.\n\
\n\
  bs=BYTES        force ibs=BYTES and obs=BYTES\n\
  cbs=BYTES       convert BYTES bytes at a time\n\
  conv=CONVS      convert the file as per the comma separated symbol list\n\
  count=SECTORS   copy only SECTORS input sectors\n\
  ibs=BYTES       read BYTES bytes at a time (must be a multiple of input sector size)\n\
"), stdout);
      fputs (_("\
  if=FILE         read from FILE instead of stdin\n\
  ifjoin=BASE.FMT   read from split files with name BASE and splitformat FMT\n\
  iflag=FLAGS     read as per the comma separated symbol list\n\
  pattern=HEX     write HEX to every byte of the output\n\
  textpattern=TEXT   write the string TEXT repeatedly to the output\n\
  obs=BYTES       write BYTES bytes at a time\n\
  of=FILE         write to FILE instead of stdout\n\
  of:=COMMAND     pipe output to the given command\n\
  oflag=FLAGS     write as per the comma separated symbol list\n\
  wipe=FILE       wipe device FILE with zeros (or specify pattern/textpattern)\n\
  seek=SECTORS    skip SECTORS input sectors at start of output\n\
  skip=SECTORS    skip SECTORS input sectors at start of input\n\
  status=noxfer   suppress transfer statistics\n\
"), stdout);
      fputs (_("\
  split=BYTES     split the output into pieces of size BYTES\n\
  splitformat=FMT   create extensions for split pieces using FMT\n\
                  Extensions can be numerical starting at zero,\n\
                  numerical starting at one, or alphabetical.\n\
                  These options are selected by using a series of\n\
                  zeros, ones, or a's, respectively. The number\n\
                  of characters used indicates the desired length of\n\
                  the extensions. For example, splitformat=1111\n\
                  indicates four character numerical extensions\n\
                  starting with 0001.\n\
  progress=on     displays a progress meter\n\
  progresscount=NUM  number of blocks processed between each progress update\n\
  sizeprobe=on    estimates size of input file for use with status\n\
  hash=ALGORITHM  computes ALGORITHM hashes of the input data\n\
  hashconv=WHEN   determines when data should be hashed, either before\n\
                  or after conversions\n\
"), stdout);
      /*
  hashformat=FORMAT         format for writing piecewise hashes\n\
  totalhashformat=FORMAT    format for writing overall hashes\n\
      */
      fputs (_("\
  hashwindow=BYTES  number of bytes for piecewise hashing\n\
  hashlog=FILE    appends piecewise hashes to the log file\n\
  errlog=FILE     appends errors to the log file\n\
  log=FILE        appends hashes and errors to the same file\n\
  errors=group    group read errors together\n\
"), stdout);  
      fputs (_("\
  vf=FILE         verify the input against FILE\n\
  vfjoin=BASE.FMT verify the input against split files with name BASE and splitformat FMT\n\
  verifylog=FILE  write the results of the verify to the given file\n\
"), stdout);

      fputs (_("\
\n\
ALGORITHM can be a comma separated list of md5, sha1, sha256, and sha512\n\
\n\
"), stdout);  
      fputs (_("\
\n\
BLOCKS and BYTES may be followed by the following multiplicative suffixes:\n\
xM M, c 1, w 2, b 512, kB 1000, K 1024, MB 1000*1000, M 1024*1024,\n\
GB 1000*1000*1000, G 1024*1024*1024, and so on for T, P, E, Z, Y.\n\
\n\
Each CONV symbol may be:\n\
\n\
"), stdout);
      fputs (_("\
  ascii     from EBCDIC to ASCII\n\
  ebcdic    from ASCII to EBCDIC\n\
  ibm       from ASCII to alternate EBCDIC\n\
  block     pad newline-terminated records with spaces to cbs-size\n\
  unblock   replace trailing spaces in cbs-size records with newline\n\
  lcase     change upper case to lower case\n\
"), stdout);
      fputs (_("\
  nocreat   do not create the output file\n\
  excl      fail if the output file already exists\n\
  notrunc   do not truncate the output file\n\
  ucase     change lower case to upper case\n\
  swab      swap every pair of input bytes\n\
"), stdout);
      fputs (_("\
  noerror   continue after read errors\n\
  sync      pad every input block with NULs to ibs-size; when used\n\
            with block or unblock, pad with spaces rather than NULs\n\
  fdatasync  physically write output file data before finishing\n\
  fsync     likewise, but also write metadata\n\
"), stdout);
      fputs (_("\
\n\
Each FLAG symbol may be:\n\
\n\
  append    append mode (makes sense only for output; conv=notrunc suggested)\n\
"), stdout);
      if (O_DIRECT)
	fputs (_("  direct    use direct I/O for data\n"), stdout);
      if (O_DIRECTORY)
	fputs (_("  directory fail unless a directory\n"), stdout);
      if (O_DSYNC)
	fputs (_("  dsync     use synchronized I/O for data\n"), stdout);
      if (O_SYNC)
	fputs (_("  sync      likewise, but also for metadata\n"), stdout);
      if (O_NONBLOCK)
	fputs (_("  nonblock  use non-blocking I/O\n"), stdout);
      if (O_NOATIME)
	fputs (_("  noatime   do not update access time\n"), stdout);
      if (O_NOCTTY)
	fputs (_("  noctty    do not assign controlling terminal from file\n"),
	       stdout);
      if (HAVE_WORKING_O_NOFOLLOW)
	fputs (_("  nofollow  do not follow symlinks\n"), stdout);
      if (O_NOLINKS)
	fputs (_("  nolinks   fail if multiply-linked\n"), stdout);
      if (O_BINARY)
	fputs (_("  binary    use binary I/O for data\n"), stdout);
      if (O_TEXT)
	fputs (_("  text      use text I/O for data\n"), stdout);

      {
	char const *siginfo_name = (SIGINFO == SIGUSR1 ? "USR1" : "INFO");
	printf (_("\
\n\
Sending a %s signal to a running `dd' process makes it\n\
print I/O statistics to standard error and then resume copying.\n\
\n\
  $ dd if=/dev/zero of=/dev/null& pid=$!\n\
  $ kill -%s $pid; sleep 1; kill $pid\n\
  18335302+0 sectors in\n\
  18335302+0 sectors out\n\
  9387674624 bytes (9.4 GB) copied, 34.6279 seconds, 271 MB/s\n\
\n\
Options are:\n\
\n\
"),
		siginfo_name, siginfo_name);
      }

      fputs (HELP_OPTION_DESCRIPTION, stdout);
      fputs (VERSION_OPTION_DESCRIPTION, stdout);
      fputs ("      --flags    display compile-time flags\n", stdout);
      emit_bug_reporting_address ();
    }
  exit (status);
}




/** Begin code copied (and modified) from ../lib/error.c **/

#define __strerror_r strerror_r

static void
print_errno_message (int errnum)
{
  char const *s;

#if defined HAVE_STRERROR_R || _LIBC
  char errbuf[1024];
# if STRERROR_R_CHAR_P || _LIBC
  s = __strerror_r (errnum, errbuf, sizeof errbuf);
# else
  if (__strerror_r (errnum, errbuf, sizeof errbuf) == 0)
    s = errbuf;
  else
    s = 0;
# endif
#else
  s = strerror (errnum);
#endif

#if !_LIBC
  if (! s)
    s = _("Unknown system error");
#endif

#if _LIBC
  __fxprintf (NULL, ": %s", s);
#else
  fprintf (stderr, ": %s", s);
  if (error_log != NULL)
    fprintf (error_log, ": %s", s);
#endif
}



static void
dc3_error_tail(int status, int errnum, const char *message, va_list args)
{
  va_list arg2;
  va_copy(arg2,args);

#if _LIBC
  if (_IO_fwide (stderr, 0) > 0)
    {
# define ALLOCA_LIMIT 2000
      size_t len = strlen (message) + 1;
      wchar_t *wmessage = NULL;
      mbstate_t st;
      size_t res;
      const char *tmp;
      bool use_malloc = false;
      
      while (1)
        {
          if (__libc_use_alloca (len * sizeof (wchar_t)))
            wmessage = (wchar_t *) alloca (len * sizeof (wchar_t));
          else
            {
              if (!use_malloc)
                wmessage = NULL;
	      
              wchar_t *p = (wchar_t *) realloc (wmessage,
                                                len * sizeof (wchar_t));
              if (p == NULL)
                {
                  free (wmessage);
                  fputws_unlocked (L"out of memory\n", stderr);
                  return;
                }
              wmessage = p;
              use_malloc = true;
            }
	  memset (&st, '\0', sizeof (st));
          tmp = message;

          res = mbsrtowcs (wmessage, &tmp, len, &st);
          if (res != len)
            break;

          if (__builtin_expect (len >= SIZE_MAX / 2, 0))
            {
              /* This really should not happen if everything is fine.  */
              res = (size_t) -1;
              break;
            }

          len *= 2;
        }

      if (res == (size_t) -1)
        {
          /* The string cannot be converted.  */
          if (use_malloc)
            {
              free (wmessage);
              use_malloc = false;
            }
          wmessage = (wchar_t *) L"???";
        }

      __vfwprintf (stderr, wmessage, args);
      if (error_log != NULL)
	__vfwprintf (error_log, wmessage, arg2);


      if (use_malloc)
        free (wmessage);
    }
  else
#endif
    {
      vfprintf (stderr, message, args);
      if (error_log != NULL)
	vfprintf (error_log, message, arg2);
    }

  va_end (args);
  
  ++error_message_count;
  if (errnum)
    print_errno_message (errnum);
#if _LIBC
  __fxprintf (NULL, "\n");
#else
  putc ('\n', stderr);
  if (error_log != NULL)
    {
      putc ('\n', error_log);
      fflush(error_log);
    }
#endif
  fflush (stderr);
  if (status)
    exit (status); 
}




static void 
dc3_error(int status, int errnum, const char *message, ...)
{
  va_list args;
  
#if defined _LIBC && defined __libc_ptf_call
  /* We do not want this call to be cut short by a thread
     cancellation.  Therefore disable cancellation for now.  */
  int state = PTHREAD_CANCEL_ENABLE;
  __libc_ptf_call (pthread_setcancelstate, (PTHREAD_CANCEL_DISABLE, &state),
                   0);
#endif
  
  fflush (stdout);
#ifdef _LIBC
  _IO_flockfile (stderr);
#endif
#if _LIBC
  __fxprintf (NULL, "%s%s: ", (dc3_progress_printed?"\n":""),program_name);
#else
  fprintf (stderr, "%s%s: ", (dc3_progress_printed?"\n":""),program_name);
#endif

  if (error_log != NULL)
    fprintf (error_log, "%s: ", program_name);

  va_start (args, message);
  dc3_error_tail(status,errnum,message,args);
  
#ifdef _LIBC
  _IO_funlockfile (stderr);
# ifdef __libc_ptf_call
  __libc_ptf_call (pthread_setcancelstate, (state, NULL), 0);
# endif
#endif
}

/** End code copied (and modified) from ../lib/error.c **/
 

/** Begin code copied (and modified) from md5deep helpers.c **/

#ifdef __linux__

#ifdef HAVE_SYS_IOCTL_H
#ifdef HAVE_SYS_MOUNT_H

# include <sys/mount.h>

#endif
#endif

static void 
dc3_probe_file(int fd)
{
  uintmax_t num_sectors = 0, sector_size = 0;
  struct stat sb;

  if (fstat(fd,&sb))
    return;

  if (S_ISREG(sb.st_mode) || S_ISDIR(sb.st_mode))
    {
      input_bytes = sb.st_size;
      return;
    }

#ifdef HAVE_SYS_IOCTL_H
#ifdef HAVE_SYS_MOUNT_H

  if (S_ISCHR(sb.st_mode) || S_ISBLK(sb.st_mode))
  {
#if defined(_IO) && defined(BLKGETSIZE)
    if (ioctl(fd, BLKGETSIZE, &num_sectors))
      return;
#else
    // If we can't run the ioctl call, we can't do anything here
    return;
#endif // ifdefined _IO and BLKGETSIZE



#if defined(_IO) && defined(BLKSSZGET)

    if (ioctl(fd, BLKSSZGET, &sector_size))
      return;


    if (sector_size != 0)
    {
      //fprintf(stderr, "dc3_probe_file: probed sector size to be %d\n", sector_size);
      dc3_sectorsize = sector_size;
    }

    if (0 == sector_size)
      sector_size = 512;
    else if (sector_size != 512)
        // fixup num_sectors if necessary
        num_sectors = num_sectors * 512 / sector_size;
#else
    sector_size = 512;
#endif  // ifdef _IO and BLKSSZGET

    input_bytes = (num_sectors * sector_size);

    //fprintf(stderr, "dc3_probe_file: input_bytes = %jd (%jd * %jd)\n", input_bytes, num_sectors, sector_size);
  }



#endif // #ifdef HAVE_SYS_MOUNT_H
#endif // #ifdef HAVE_SYS_IOCTL_H
}  


static void 
dc3_probe_sectorsize(int fd)
{
  uintmax_t sector_size = 0;
  struct stat sb;

  if (fstat(fd,&sb))
    return;

  if (S_ISREG(sb.st_mode) || S_ISDIR(sb.st_mode))
    {
      input_bytes = sb.st_size;
      return;
    }

#ifdef HAVE_SYS_IOCTL_H
#ifdef HAVE_SYS_MOUNT_H

  if (S_ISCHR(sb.st_mode) || S_ISBLK(sb.st_mode))
  {
#if defined(_IO) && defined(BLKSSZGET)

    if (ioctl(fd, BLKSSZGET, &sector_size))
      return;

    if (sector_size != 0)
    {
      //fprintf(stderr, "dc3_probe_sectorsize: probed sector size to be %d\n", sector_size);
      dc3_sectorsize = sector_size;
    }

#endif  // ifdef _IO and BLKSSZGET
  }

#endif // #ifdef HAVE_SYS_MOUNT_H
#endif // #ifdef HAVE_SYS_IOCTL_H
}



#elif defined (__APPLE__)

#ifdef HAVE_SYS_IOCTL_H
# include <sys/disk.h>
# include <sys/ioctl.h>
#endif

static void 
dc3_probe_file(int fd)
{
  struct stat info;
  off_t total = 0;
  /* RBF - Is there a better way to emulate ftell? */
  off_t original = lseek(fd,0,SEEK_CUR);
  uint32_t blocksize = 0;
  uint64_t blockcount = 0;

  /* I'd prefer not to use fstat as it will follow symbolic links. We don't
     follow symbolic links. That being said, all symbolic links *should*
     have been caught before we got here. */

  if (fstat(fd, &info))
    return;

#ifdef HAVE_SYS_IOCTL_H
  /* Block devices, like /dev/hda, don't return a normal filesize.
     If we are working with a block device, we have to ask the operating
     system to tell us the true size of the device. 
     
     This isn't the recommended way to do check for block devices, 
     but using S_ISBLK(info.stmode) wasn't working. */
  if (info.st_mode & S_IFBLK)
  {    
    /* Get the block size */
    if (ioctl(fd, DKIOCGETBLOCKSIZE,&blocksize) < 0) 
      return;

  /* Get the number of blocks */
    if (ioctl(fd, DKIOCGETBLOCKCOUNT, &blockcount) < 0) 
      return;
 
    if (blocksize != 0)
    {
      //fprintf(stderr, "dc3_probe_file: probed sector size to be %d\n", blocksize);
      dc3_sectorsize = blocksize;
    }

    total = blocksize * blockcount;
  }
#endif     // ifdef HAVE_IOCTL_H

  else 
  {
    total = lseek(fd,0,SEEK_END);
    if ((lseek(fd,original,SEEK_SET)))
      return;
  }

  input_bytes = (total - original);
}


static void 
dc3_probe_sectorsize(int fd)
{
  struct stat info;
  uint32_t blocksize = 0;

  /* I'd prefer not to use fstat as it will follow symbolic links. We don't
     follow symbolic links. That being said, all symbolic links *should*
     have been caught before we got here. */

  if (fstat(fd, &info))
    return;

#ifdef HAVE_SYS_IOCTL_H
  /* Block devices, like /dev/hda, don't return a normal filesize.
     If we are working with a block device, we have to ask the operating
     system to tell us the true size of the device. 
     
     This isn't the recommended way to do check for block devices, 
     but using S_ISBLK(info.stmode) wasn't working. */
  if (info.st_mode & S_IFBLK)
  {    
    /* Get the block size */
    if (ioctl(fd, DKIOCGETBLOCKSIZE,&blocksize) < 0) 
      return;
  
    if (blocksize != 0)
    {
      //fprintf(stderr, "dc3_probe_sectorsize: probed sector size to be %d\n", blocksize);
      dc3_sectorsize = blocksize;
    }
    //else { fprintf(stderr, "dc3_probe_sectorsize: failed to probe sector size\n"); }
  }
#endif     // ifdef HAVE_IOCTL_H
}

#elif defined (__CYGWIN__) // ifdef __APPLE__

#include <sys/ioctl.h>
#include <cygwin/types.h>
#include <cygwin/fs.h>


static void 
dc3_probe_file(int fd)
{
  uintmax_t num_sectors = 0, sector_size = 0;
  struct stat sb;

  if (fstat(fd,&sb))
    return;

  if (S_ISREG(sb.st_mode) || S_ISDIR(sb.st_mode))
    {
      input_bytes = sb.st_size;
      return;
    }

  if (S_ISCHR(sb.st_mode) || S_ISBLK(sb.st_mode))
  {
    if (ioctl(fd, BLKGETSIZE, &num_sectors))
      return;

    if (ioctl(fd, BLKSSZGET, &sector_size))
      return;

    if (sector_size != 0)
      dc3_sectorsize = sector_size;

    if (0 == sector_size)
      sector_size = 512;
    else if (sector_size != 512) // fixup num_sectors if necessary
        num_sectors = num_sectors * 512 / sector_size;

    input_bytes = (num_sectors * sector_size);

    //fprintf(stderr, "dc3_probe_file: input_bytes = %jd (%jd * %jd)\n",
    //        input_bytes, num_sectors, sector_size);
  }
}  

static void 
dc3_probe_sectorsize(int fd)
{
  uintmax_t sector_size = 0;
  struct stat sb;

  if (fstat(fd,&sb))
    return;

  if (S_ISREG(sb.st_mode) || S_ISDIR(sb.st_mode))
    {
      input_bytes = sb.st_size;
      return;
    }

  if (S_ISCHR(sb.st_mode) || S_ISBLK(sb.st_mode))
  {
    if (ioctl(fd, BLKSSZGET, &sector_size))
      return;

    if (sector_size != 0)
      dc3_sectorsize = sector_size;

    //fprintf(stderr, "dc3_probe_sectorsize: probed sector size to be %d\n", sector_size);
  }
}

#else // ifdef __CYGWIN__ 

/* This is code for general UNIX systems 
   (e.g. NetBSD, FreeBSD, OpenBSD, etc) */

static off_t
midpoint (off_t a, off_t b, long blksize)
{
  off_t aprime = a / blksize;
  off_t bprime = b / blksize;
  off_t c, cprime;

  cprime = (bprime - aprime) / 2 + aprime;
  c = cprime * blksize;

  return c;
}


off_t find_dev_size(int fd, int blk_size)
{

  off_t curr = 0, amount = 0;
  void *buf;
  
  if (blk_size == 0)
    return 0;
  
  buf = malloc(blk_size);
  
  for (;;) {
    ssize_t nread;
    
    lseek(fd, curr, SEEK_SET);
    nread = read(fd, buf, blk_size);
    if (nread < blk_size) 
    {
      if (nread <= 0) 
        {
          if (curr == amount) 
          {
            free(buf);
            lseek(fd, 0, SEEK_SET);
            return amount;
          }
          curr = midpoint(amount, curr, blk_size);
        }
      else 
        { /* 0 < nread < blk_size */
          free(buf);
          lseek(fd, 0, SEEK_SET);
          return amount + nread;
        }
    } 
    else 
    {
      amount = curr + blk_size;
      curr = amount * 2;
    }
  }

  free(buf);
  lseek(fd, 0, SEEK_SET);
  return amount;
}


static void 
dc3_probe_file(int fd)
{
  struct stat sb;
  
  if (fstat(fd,&sb))
    return;

  if (S_ISREG(sb.st_mode) || S_ISDIR(sb.st_mode))
    input_bytes = sb.st_size;
  else if (S_ISCHR(sb.st_mode) || S_ISBLK(sb.st_mode))
    input_bytes = find_dev_size(fd,sb.st_blksize);
}  

#endif // ifdef __LINUX__


/** End code copied (and modified) from md5deep helpers.c **/


static void
dc3_reprobe()
{
    uintmax_t save = input_bytes;
    input_bytes = 0;
    dc3_probe_file(STDIN_FILENO);
    input_bytes += save;
}


static void
translate_charset (char const *new_trans)
{
  int i;

  for (i = 0; i < 256; i++)
    trans_table[i] = new_trans[trans_table[i]];
  translation_needed = true;
}

/* Return true if I has more than one bit set.  I must be nonnegative.  */

static inline bool
multiple_bits_set (int i)
{
  return (i & (i - 1)) != 0;
}

/* Print transfer statistics.  */
static void dc3_display_progress(int print_to_log);

/* Options we use repeatedly whend displaying units */ 
static int dc3_human_opts = \
    (human_autoscale | human_round_to_nearest | human_suppress_point_zero 
     | human_space_before_unit | human_SI | human_base_1024);

static void
print_stats (void)
{
  fprintf (stderr,
	   _("%"PRIuMAX"+%"PRIuMAX" sectors in\n"
	     "%"PRIuMAX"+%"PRIuMAX" sectors out\n"),
	   dc3_sectors_r_full, dc3_sectors_r_partial,
           dc3_sectors_w_full, dc3_sectors_w_partial);

  if (error_log != NULL)
    fprintf (error_log,
	     _("%"PRIuMAX"+%"PRIuMAX" sectors in\n"
	       "%"PRIuMAX"+%"PRIuMAX" sectors out\n"),
	     dc3_sectors_r_full, dc3_sectors_r_partial,
             dc3_sectors_w_full, dc3_sectors_w_partial);

  if (r_truncate != 0)
    {
      fprintf (stderr,
	       ngettext ("%"PRIuMAX" truncated record\n",
			 "%"PRIuMAX" truncated records\n",
			 select_plural (r_truncate)),
	       r_truncate);
      if (error_log != NULL)
	fprintf (error_log,
		 ngettext ("%"PRIuMAX" truncated record\n",
			   "%"PRIuMAX" truncated records\n",
			   select_plural (r_truncate)),
		 r_truncate);
    }

  if (status_flags & STATUS_NOXFER)
    return;

  dc3_display_progress(true);

  /* Added to clear line now that dc3_display_progress uses \r */
  fprintf(stderr,_("\n"));
}


/* This function used to be part of print_stats. We need
   to print this information as a progress meter, so I
   moved it to a separate function - J Kornblum */
static void dc3_display_progress(int print_to_log)
{
  char hbuf[LONGEST_HUMAN_READABLE + 1];
  int human_opts = dc3_human_opts;
  xtime_t now = gethrxtime ();
  double delta_s;
  char const *bytes_per_second;
  float percent_done;
  
  /* Use integer arithmetic to compute the transfer rate,
     since that makes it easy to use SI abbreviations.  */

  const char* verb = NULL;
  if (dc3_mode_verify)
      verb = "compared";
  else if (dc3_mode_wipe)
      verb = "wiped";
  else
      verb = "copied";

  fprintf (stderr,
	   ngettext ("%"PRIuMAX" byte (%s) %s",
		     "%"PRIuMAX" bytes (%s) %s",
		     select_plural (w_bytes)),
	   w_bytes,
	   human_readable (w_bytes, hbuf, human_opts, 1, 1),
           verb);
  if (error_log != NULL && print_to_log)
    fprintf (error_log,
	     ngettext ("%"PRIuMAX" byte (%s) %s",
		       "%"PRIuMAX" bytes (%s) %s",
		       select_plural (w_bytes)),
	     w_bytes,
	     human_readable (w_bytes, hbuf, human_opts, 1, 1),
             verb);

  if (input_bytes != 0)
    {
      percent_done = 100 * (w_bytes / (float)input_bytes);

      fprintf (stderr," (%2.0f%%)", percent_done);
      if (error_log != NULL && print_to_log)
          fprintf (error_log," (%2.0f%%)", percent_done);
    }
  else
    {
      fprintf (stderr," (??%%)");
      if (error_log != NULL && print_to_log)
          fprintf (error_log," (??%%)");
    }

  if (start_time < now)
    {
      double XTIME_PRECISIONe0 = XTIME_PRECISION;
      uintmax_t delta_xtime = now;
      delta_xtime -= start_time;
      delta_s = delta_xtime / XTIME_PRECISIONe0;
      bytes_per_second = human_readable (w_bytes, hbuf, human_opts,
					 XTIME_PRECISION, delta_xtime);
    }
  else
    {
      delta_s = 0;
      bytes_per_second = _("Infinity B");
    }

  /* TRANSLATORS: The two instances of "s" in this string are the SI
     symbol "s" (meaning second), and should not be translated.

     This format used to be:

     ngettext (", %g second, %s/s\n", ", %g seconds, %s/s\n", delta_s == 1)

     but that was incorrect for languages like Polish.  To fix this
     bug we now use SI symbols even though they're a bit more
     confusing in English.  */
  fprintf (stderr, _(", %g s, %s/s        "), delta_s, bytes_per_second);
  fprintf (stderr,"\r");
  if (error_log != NULL && print_to_log)
    fprintf (error_log, _(", %g s, %s/s\n"), delta_s, bytes_per_second);
}

int hex2char(char *hstr)
{
    int retval;
    
    if (strlen(hstr) != 2)
        return -1;
    if (EOF == sscanf(hstr, "%x", &retval))
        return -1;
    return retval;
}


char * make_pattern(const char *pattern)
{
  size_t plen, numbytes, i;
  char * buffer;

  plen = strlen(pattern);
  if (plen == 0 || plen % 2 != 0)
    return NULL;
  
  numbytes = plen / 2;
  buffer = malloc(numbytes);
  
  for (i = 0; i < numbytes; i++) {
    char tmpstring[3];
    int byteval;
    strncpy(tmpstring, &pattern[i*2], 2);
    tmpstring[2] = '\0';
    byteval = hex2char(tmpstring);
    
    if (byteval == -1) {
      free(buffer);
      return NULL;
    }
    buffer[i] = (char)byteval;
  }

  dc3_pattern_len = numbytes;
  
  return buffer;
}


static void
cleanup (void)
{
  if (dc3_progress)
    {
      fprintf(stderr,"                                                                   \r");
    }

  if (dc3_ifjoin_done == false && close (STDIN_FILENO) < 0)
    dc3_error (EXIT_FAILURE, errno,
	   _("closing input file %s"), quote (input_file));

  /* Don't remove this call to close, even though close_stdout
     closes standard output.  This close is necessary when cleanup
     is called as part of a signal handler.  */
  if (close (STDOUT_FILENO) < 0)
    dc3_error (EXIT_FAILURE, errno,
	   _("closing output file %s"), quote (output_file));
}

static inline void ATTRIBUTE_NORETURN
quit (int code)
{
  cleanup ();

  if (!dc3_mode_verify)
    print_stats ();

  if (code != EXIT_SUCCESS)
      dc3_exit_reason = DC3_EXIT_FAILED;

  dc3_exit_msg();

  if (hash_log)
    fclose(hash_log);

  process_signals ();
  exit (code);
}

void dc3_exit_msg()
{
  // print hash so far if we were interrupted
  if (dc3_hash && !dc3_mode_verify && interrupt_signal)
  {
    dc3_hash_final();
  }

  const char* verb = NULL;
  switch (dc3_exit_reason)
  {
      case DC3_EXIT_ABORTED:
          verb = "aborted";
          break;
      case DC3_EXIT_FAILED:
          verb = "failed";
          break;
      default:
          verb = "completed";
          break;
  }

  char* dc3_stop_time = dc3_time();
  fprintf(stderr, "dc3dd %s at %s\n", verb, dc3_stop_time);
  if (error_log != NULL)
    fprintf(error_log, "dc3dd %s at %s\n", verb, dc3_stop_time);

  if (error_log != NULL)
    fflush(error_log);
}

/* An ordinary signal was received; arrange for the program to exit.  */

static void
interrupt_handler (int sig)
{
  if (! SA_RESETHAND)
    signal (sig, SIG_DFL);
  interrupt_signal = sig;
}

/* An info signal was received; arrange for the program to print status.  */

static void
siginfo_handler (int sig)
{
  if (! SA_NOCLDSTOP)
    signal (sig, siginfo_handler);
  info_signal_count++;
}

/* Install the signal handlers.  */

static void
install_signal_handlers (void)
{
  bool catch_siginfo = ! (SIGINFO == SIGUSR1 && getenv ("POSIXLY_CORRECT"));

#if SA_NOCLDSTOP

  struct sigaction act;
  sigemptyset (&caught_signals);
  if (catch_siginfo)
    {
      sigaction (SIGINFO, NULL, &act);
      if (act.sa_handler != SIG_IGN)
	sigaddset (&caught_signals, SIGINFO);
    }
  sigaction (SIGINT, NULL, &act);
  if (act.sa_handler != SIG_IGN)
    sigaddset (&caught_signals, SIGINT);
  act.sa_mask = caught_signals;

  if (sigismember (&caught_signals, SIGINFO))
    {
      act.sa_handler = siginfo_handler;
      act.sa_flags = 0;
      sigaction (SIGINFO, &act, NULL);
    }

  if (sigismember (&caught_signals, SIGINT))
    {
      /* POSIX 1003.1-2001 says SA_RESETHAND implies SA_NODEFER,
	 but this is not true on Solaris 8 at least.  It doesn't
	 hurt to use SA_NODEFER here, so leave it in.  */
      act.sa_handler = interrupt_handler;
      act.sa_flags = SA_NODEFER | SA_RESETHAND;
      sigaction (SIGINT, &act, NULL);
    }

#else

  if (catch_siginfo && signal (SIGINFO, SIG_IGN) != SIG_IGN)
    {
      signal (SIGINFO, siginfo_handler);
      siginterrupt (SIGINFO, 1);
    }
  if (signal (SIGINT, SIG_IGN) != SIG_IGN)
    {
      signal (SIGINT, interrupt_handler);
      siginterrupt (SIGINT, 1);
    }
#endif
}

/* Process any pending signals.  If signals are caught, this function
   should be called periodically.  Ideally there should never be an
   unbounded amount of time when signals are not being processed.  */

static void
process_signals (void)
{
  while (interrupt_signal | info_signal_count)
    {
      int interrupt;
      int infos;
      sigset_t oldset;

      sigprocmask (SIG_BLOCK, &caught_signals, &oldset);

      /* Reload interrupt_signal and info_signal_count, in case a new
	 signal was handled before sigprocmask took effect.  */
      interrupt = interrupt_signal;
      infos = info_signal_count;

      if (infos)
	info_signal_count = infos - 1;

      sigprocmask (SIG_SETMASK, &oldset, NULL);

      if (interrupt)
	cleanup ();
      print_stats ();
      if (interrupt)
      {
        dc3_exit_reason = DC3_EXIT_ABORTED;
	dc3_exit_msg();
	raise (interrupt);
      }
    }
}

/* Read from FD into the buffer BUF of size SIZE, processing any
   signals that arrive before bytes are read.  Return the number of
   bytes read if successful, -1 (setting errno) on failure.  */

static ssize_t
iread (int fd, char *buf, size_t size)
{
  for (;;)
    {
      ssize_t nread;
      process_signals ();
      nread = read (fd, buf, size);
      if (! (nread < 0 && errno == EINTR))
	return nread;
    }
}

/* Write to FD the buffer BUF of size SIZE, processing any signals
   that arrive.  Return the number of bytes written, setting errno if
   this is less than SIZE.  Keep trying if there are partial
   writes.  */

static size_t
dc3_iwrite (int fd, char const *buf, size_t size)
{
  size_t total_written = 0;

  while (total_written < size)
    {
      ssize_t nwritten;
      process_signals ();
      nwritten = write (fd, buf + total_written, size - total_written);
      if (nwritten < 0)
	{
	  if (errno != EINTR)
	    break;
	}
      else if (nwritten == 0)
	{
	  /* Some buggy drivers return 0 when one tries to write beyond
	     a device's end.  (Example: Linux 1.2.13 on /dev/fd0.)
	     Set errno to ENOSPC so they get a sensible diagnostic.  */
	  errno = ENOSPC;
	  break;
	}
      else
	total_written += nwritten;
    }

  return total_written;
}

static char *dc3_split_numbers = "0123456789";
#define DC3_SPLIT_NUM_NUMBERS  10
static char *dc3_split_letters = "abcdefghijklmnopqrstuvwxyz";
#define DC3_SPLIT_NUM_LETTERS  26

static int
dc3_generate_split_filename(const char* base, const char* fmt, uintmax_t n, char** result)
{
  /* The extra two bytes accounts for the dot between the filename
     and the extension and then for the trailing \0 */
  size_t fmt_len = strlen(fmt);
  size_t base_len = strlen(base);
  size_t len  = base_len + fmt_len + 2;

  if (NULL == *result)
    {
      *result = (char *)malloc(len);
      if (NULL == *result)
	dc3_error(EXIT_FAILURE, 0, _("Unable to allocate filename"));
    }

  snprintf(*result, base_len+2, "%s.", base);

  /* In case we're starting with .001 */
  uintmax_t num = n + (fmt[0] == '1' ? 1 : 0);
  
  int i;
  for (i = fmt_len - 1; i >= 0 ; i--)
    {
      uintmax_t x;

      if ('a' == fmt[0])
	{
	  x = num % DC3_SPLIT_NUM_LETTERS;
          (*result)[base_len+1+i] = dc3_split_letters[x];
	  num /= DC3_SPLIT_NUM_LETTERS;
	}
      else
	{
	  x = num % DC3_SPLIT_NUM_NUMBERS;
	  (*result)[base_len+1+i] = dc3_split_numbers[x];
	  num /= DC3_SPLIT_NUM_NUMBERS;
	}
    }
  
  (*result)[len-1] = 0;

  if (num > 0)
    dc3_error(EXIT_FAILURE,0,_("Split extensions exhausted"));

  return false;
}


static void dc3_open_split(void)
{
  close(STDOUT_FILENO);

  dc3_split_number++;
  dc3_generate_split_filename(output_file, dc3_split_format, dc3_split_number, &dc3_split_filename);

  mode_t perms = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH;
  int opts
    = (output_flags
       | (conversions_mask & C_NOCREAT ? 0 : O_CREAT)
       | (conversions_mask & C_EXCL ? O_EXCL : 0)
       | (seek_sectors || (conversions_mask & C_NOTRUNC) ? 0 : O_TRUNC));
  
  if (fd_reopen(STDOUT_FILENO, dc3_split_filename, O_WRONLY | opts, perms) < 0)
    dc3_error(EXIT_FAILURE, errno, _("opening %s"), quote(dc3_split_filename));

  dc3_split_current = 0;
}


static size_t
dc3_verify (int fd, char const *src_buf, size_t size)
{
  char * dest_buf;

  dest_buf = (char *)malloc(size);
  if (NULL == dest_buf)
    dc3_error(EXIT_FAILURE,errno,_("Unable to allocate memory"));

  ssize_t n = iread(fd,dest_buf,size);

  // TODO: could enable block below to give better reporting of the
  //       reason why a verify failed (was it input or output that couldn't be read)
  /*
  if (n < 0)
      fprintf(stderr, "unrecoverable error reading verify file %s\n", output_file);
  else*/ if (dc3_vfjoin && 0 <= n && n < size)
  {
      //fprintf(stderr, "opening next vfjoin piece\n");
      dc3_vfjoin_number++;
      dc3_generate_split_filename(dc3_vfjoin_base, dc3_vfjoin_ext, dc3_vfjoin_number, &output_file);

      if (fd_reopen(STDOUT_FILENO, output_file, O_RDONLY, 0) < 0)
          dc3_error(EXIT_FAILURE, 0, _("Verify FAILED"));
      else
         iread(fd, dest_buf+n, size-n);
  }
 
  if (memcmp(dest_buf,src_buf,size))
    dc3_error(EXIT_FAILURE,0,_("Verify FAILED"));

  free(dest_buf);
  return size;
}

	   


static size_t
iwrite (int fd, char const *buf, size_t size)
{
  if (dc3_mode_verify)
    return dc3_verify(fd,buf,size);

  if (! dc3_split )
    return dc3_iwrite(fd,buf,size);

  uintmax_t left = dc3_split_bytes - dc3_split_current;
  size_t nwritten;

  if (0 == left)
    {
      dc3_open_split();
      left = dc3_split_bytes;
    }

  if (size <= left)
    {
      nwritten           = dc3_iwrite(fd,buf,size);
      dc3_split_current += nwritten;
    }
  else
    {
      nwritten = dc3_iwrite(fd,buf,left);
      dc3_split_current += nwritten;
      nwritten = dc3_iwrite(fd,buf+nwritten,size-nwritten);
      dc3_split_current += nwritten;
    }

  return nwritten;
}

void
dc3_write_stats(size_t n)
{
    if (n <= 0)
        return;

    w_bytes += n;

    dc3_sectors_w_full += n / dc3_sectorsize;

    if (n % dc3_sectorsize)
        dc3_sectors_w_partial++;
}

/* Write, then empty, the output buffer `obuf'. */

static void
write_output (void)
{
  size_t nwritten = iwrite (STDOUT_FILENO, obuf, output_blocksize);
  dc3_write_stats(nwritten);

  //fprintf(stderr, "write_output: wrote %zd bytes, total %zd\n", nwritten, w_bytes);
  if (nwritten != output_blocksize)
  {
      if (dc3_mode_wipe && errno == ENOSPC)
      {
          quit (EXIT_SUCCESS);
      }
      else
      {
          dc3_error (0, errno, _("writing to %s"), quote (output_file));
          quit (EXIT_FAILURE);
      }
  }

  //fprintf(stderr, "write_output: sector count now %jd + %jd\n", dc3_sectors_w_full, dc3_sectors_w_partial);
  oc = 0;
}

/* Return true if STR is of the form "PATTERN" or "PATTERNDELIM...".  */

static bool
operand_matches (char const *str, char const *pattern, char delim)
{
  while (*pattern)
    if (*str++ != *pattern++)
      return false;
  return !*str || *str == delim;
}

/* Interpret one "conv=..." or similar operand STR according to the
   symbols in TABLE, returning the flags specified.  If the operand
   cannot be parsed, use ERROR_MSGID to generate a diagnostic.  */

static int
parse_symbols (char const *str, struct symbol_value const *table,
	       char const *error_msgid)
{
  int value = 0;

  for (;;)
    {
      char const *strcomma = strchr (str, ',');
      struct symbol_value const *entry;

      for (entry = table;
	   ! (operand_matches (str, entry->symbol, ',') && entry->value);
	   entry++)
	{
	  if (! entry->symbol[0])
	    {
	      size_t slen = strcomma ? strcomma - str : strlen (str);
	      dc3_error (0, 0, "%s: %s", _(error_msgid),
		     quotearg_n_style_mem (0, locale_quoting_style, str, slen));

	      usage (EXIT_FAILURE);
	    }
	}

      value |= entry->value;
      if (!strcomma)
	break;
      str = strcomma + 1;

    }

  return value;
}

/* Return the value of STR, interpreted as a non-negative decimal integer,
   optionally multiplied by various values.
   Set *INVALID if STR does not represent a number in this format.  */

static uintmax_t
parse_integer (const char *str, bool *invalid)
{
  uintmax_t n;
  char *suffix;
  enum strtol_error e = xstrtoumax (str, &suffix, 10, &n, "bcEGkKMPTwYZ0");

  if (e == LONGINT_INVALID_SUFFIX_CHAR && *suffix == 'x')
    {
      uintmax_t multiplier = parse_integer (suffix + 1, invalid);

      if (multiplier != 0 && n * multiplier / multiplier != n)
	{
	  *invalid = true;
	  return 0;
	}

      n *= multiplier;
    }
  else if (e != LONGINT_OK)
    {
      *invalid = true;
      return 0;
    }

  return n;
}

/* OPERAND is of the form "X=...".  Return true if X is NAME.  */

static bool
operand_is (char const *operand, char const *name)
{
  return operand_matches (operand, name, '=');
}

static void
scanargs (int argc, char *const *argv)
{
  int i;
  size_t blocksize = 0;

  for (i = optind; i < argc; i++)
    {
      char const *name = argv[i];
      char const *val = strchr (name, '=');

      if (val == NULL)
	{
	  dc3_error (0, 0, _("unrecognized operand %s"), quote (name));
	  usage (EXIT_FAILURE);
	}
      val++;

      if (operand_is (name,"pattern"))
	{
	  dc3_use_pattern   = true;
	  dc3_pattern_value = make_pattern(val);
	  if (NULL == dc3_pattern_value)
	    dc3_error(EXIT_FAILURE, 0, _("illegal pattern %s"), quote(val));
	}

      else if (operand_is (name,"textpattern"))
	{
	  dc3_pattern_value = strdup(val);
	  if (NULL == dc3_pattern_value)
	    dc3_error(EXIT_FAILURE, 0, _("illegal pattern %s"), quote(val));
	  dc3_use_pattern   = true;
	  dc3_pattern_len   = strlen(dc3_pattern_value);
	}

      else if (operand_is (name,"errlog"))
	{
	  error_log = fopen(val,"a");
	  if (NULL == error_log)
	    dc3_error (EXIT_FAILURE, errno, _("opening %s"), quote (val));
	}

      else if (operand_is (name,"hashlog"))
	{
	  hash_log = fopen(val,"a");
	  if (NULL == hash_log)
	    dc3_error (EXIT_FAILURE, errno, _("opening %s"), quote (val));
	  dc3_hash = true;
	}

      else if (operand_is (name,"log"))
	{
	  hash_log = fopen(val,"a");
	  if (NULL == hash_log)
	    dc3_error (EXIT_FAILURE, errno, _("opening %s"), quote (val));
	  error_log = hash_log;
	  dc3_hash = true;
	}
      else if (operand_is (name,"errors"))
	{
	  dc3_group_errors = true;
	}
	

      else if (operand_is (name,"hash"))
	{
	  dc3_parse_hash(val);
	  dc3_hash = true;
	  if (NULL == hash_log)
	    hash_log = stderr;
	}

      else if (operand_is (name,"hashconv"))
	{
	  if (STREQ (val, "before"))
	    dc3_hash_before = true;
	  else if (STREQ (val, "after"))
	    dc3_hash_before = false;
	  else
	    dc3_error (EXIT_FAILURE, 0, _("unknown hash convention %s"), 
		   quote(val));
	}
      
      /*
      else if (operand_is (name,"hashformat"))
	dc3_error (EXIT_FAILURE, 0, _("not implemented yet %s"), quote(name));
      else if (operand_is (name,"totalhashformat"))
	dc3_error (EXIT_FAILURE, 0, _("not implemented yet %s"), quote(name));
      else if (operand_is (name,"rate"))
	dc3_error (0, 0, _("%s is not implemented, ignored"), quote(name));
      */
     

      else if (operand_is (name,"progress"))
	dc3_progress = true;

      else if (operand_is (name,"sizeprobe"))
	dc3_sizeprobe = true;

      else if (operand_is (name,"zork"))
	dc3_error (EXIT_FAILURE, 0, _("It is pitch dark here. You are likely to be eaten by a grue."));

      else if (operand_is (name,"splitformat"))
	{
          if (!dc3_validate_splitformat(val))
            dc3_error(EXIT_FAILURE, 0, _("Illegal split format %s"), quote(val));

	  dc3_split_format = val;
	  if (!dc3_split)
	    {
	      dc3_split       = true;
	      dc3_split_bytes = blocksize;
	    }
	}

      else if (operand_is(name, "ifjoin"))
        {
          const char* base = dc3_filebase(val, ".");

          if (base == NULL)
            dc3_error(EXIT_FAILURE, 0, _("Illegal ifjoin format %s - missing extension"), quote(val));

          const char* ext  = dc3_fileext(val, ".");

          if (strlen(ext) == 0 || !dc3_validate_splitformat(ext))
            dc3_error(EXIT_FAILURE, 0, _("Illegal ifjoin format %s"), quote(ext));
         
          dc3_ifjoin_base = strdup(base);
          dc3_ifjoin_ext  = strdup(ext);
          dc3_ifjoin      = true;

          conversions_mask |= C_TWOBUFS;
        }

      else if (operand_is (name,"vf"))
	{
	  output_file = val;
	  dc3_mode_verify = true;
	}

      else if (operand_is(name, "vfjoin"))
        {
          const char* base = dc3_filebase(val, ".");

          if (base == NULL)
            dc3_error(EXIT_FAILURE, 0, _("Illegal vfjoin format %s - missing extension"), quote(val));

          const char* ext  = dc3_fileext(val, ".");

          if (strlen(ext) == 0 || !dc3_validate_splitformat(ext))
            dc3_error(EXIT_FAILURE, 0, _("Illegal vfjoin format %s"), quote(ext));

          dc3_vfjoin_base = strdup(base);
          dc3_vfjoin_ext  = strdup(ext);
          dc3_vfjoin      = true;
        }

      else if (operand_is (name,"verifylog"))
	{
	  error_log = fopen(val,"a");
	  if (NULL == error_log)
	    dc3_error (EXIT_FAILURE, errno, _("opening %s"), quote (val));
	}

      else if (operand_is (name,"verifylog:"))
	dc3_error (EXIT_FAILURE, 0, _("%s not implemented yet"), quote(name));

      else if (operand_is (name,"of:"))
	{
	  output_file = val;
	  dc3_pipe_output = true;
	}
      
      else if (operand_is (name, "if"))
	{
	  /* To save I/O time, we replace if=/dev/zero with pattern=00 */
	  if (STREQ (val,"/dev/zero"))
	    {
	      dc3_use_pattern   = true;
	      dc3_pattern_value = make_pattern("00");
	      if (NULL == dc3_pattern_value)
		dc3_error(EXIT_FAILURE,0, _("illegal pattern %s"), quote(val));
	    }
	  else
	    input_file = val;
	}
      else if (operand_is (name, "of"))
	output_file = val;
      else if (operand_is (name, "wipe"))
      {
	  // wipe a device:
	  // use pattern=00 [unless overridden]
	  // use count=(sizeprobe output_device) [unless overridden]
	  // if, for some reason, sizeprobe is unavailable, run until end of device reported (ENOSPC) and report completion
	  output_file = val;
	  dc3_mode_wipe = true;
      }
      else if (operand_is (name, "conv"))
	conversions_mask |= parse_symbols (val, conversions,
					   N_("invalid conversion"));
      else if (operand_is (name, "iflag"))
	input_flags |= parse_symbols (val, flags,
				      N_("invalid input flag"));
      else if (operand_is (name, "oflag"))
	output_flags |= parse_symbols (val, flags,
				       N_("invalid output flag"));
      else if (operand_is (name, "status"))
	status_flags |= parse_symbols (val, statuses,
				       N_("invalid status flag"));
      else
	{
	  bool invalid = false;
	  uintmax_t n = parse_integer (val, &invalid);

	  if (operand_is (name,"hashwindow"))
	    {
	      invalid |= ( 0 == n);
	      dc3_hash_window = n;
	      dc3_hash        = true;
	    }

	  else if (operand_is (name,"progresscount"))
	    {
	      invalid |= (0 == n);
	      dc3_progress_count = n;
	      dc3_progress = true;
	    }

	  else if (operand_is (name,"split"))
	    {
	      invalid |= ( 0 == n);
	      dc3_split_bytes = n;
	      dc3_split = true;
	    }

	  else if (operand_is (name, "ibs"))
	    {
	      invalid |= ! (0 < n && n <= MAX_BLOCKSIZE (INPUT_BLOCK_SLOP));
	      input_blocksize = n;
	      conversions_mask |= C_TWOBUFS;
	    }
	  else if (operand_is (name, "obs"))
	    {
	      invalid |= ! (0 < n && n <= MAX_BLOCKSIZE (OUTPUT_BLOCK_SLOP));
	      output_blocksize = n;
	      conversions_mask |= C_TWOBUFS;
	    }
	  else if (operand_is (name, "bs"))
	    {
	      invalid |= ! (0 < n && n <= MAX_BLOCKSIZE (INPUT_BLOCK_SLOP));
	      blocksize = n;
	    }
	  else if (operand_is (name, "cbs"))
	    {
	      invalid |= ! (0 < n && n <= SIZE_MAX);
	      conversion_blocksize = n;
	    }
	  else if (operand_is (name, "skip"))
            skip_sectors = n;
	  else if (operand_is (name, "seek"))
            seek_sectors = n;
	  else if (operand_is (name, "count"))
            max_sectors = n;
	  else
	    {
 	      dc3_error (0, 0, _("unrecognized operand %s"), quote (name));
	      usage (EXIT_FAILURE);
	    }

	  if (invalid)
	    dc3_error (EXIT_FAILURE, 0, _("invalid number %s"), quote (val));
	}
    }

  if (blocksize)
    input_blocksize = output_blocksize = blocksize;

  /* If bs= was given, both `input_blocksize' and `output_blocksize' will
     have been set to positive values.  If either has not been set,
     bs= was not given, so make sure two buffers are used. */
  if (input_blocksize == 0 || output_blocksize == 0)
    conversions_mask |= C_TWOBUFS;
  if (input_blocksize == 0)
    input_blocksize = DEFAULT_BLOCKSIZE;
  if (output_blocksize == 0)
    output_blocksize = DEFAULT_BLOCKSIZE;
  if (conversion_blocksize == 0)
    conversions_mask &= ~(C_BLOCK | C_UNBLOCK);

  // automatically enable "dynamic" error recovery when conv=noerror,sync specified
  if (conversions_mask & (C_SYNC | C_NOERROR))
    conversions_mask |= C_DYNAMIC;

  if (input_flags & (O_DSYNC | O_SYNC))
    input_flags |= O_RSYNC;

  if (multiple_bits_set (conversions_mask & (C_ASCII | C_EBCDIC | C_IBM)))
    dc3_error (EXIT_FAILURE, 0, _("cannot combine any two of {ascii,ebcdic,ibm}"));
  if (multiple_bits_set (conversions_mask & (C_BLOCK | C_UNBLOCK)))
    dc3_error (EXIT_FAILURE, 0, _("cannot combine block and unblock"));
  if (multiple_bits_set (conversions_mask & (C_LCASE | C_UCASE)))
    dc3_error (EXIT_FAILURE, 0, _("cannot combine lcase and ucase"));
  if (multiple_bits_set (conversions_mask & (C_EXCL | C_NOCREAT)))
    dc3_error (EXIT_FAILURE, 0, _("cannot combine excl and nocreat"));

  if (input_file != NULL && dc3_ifjoin)
    dc3_error (EXIT_FAILURE, 0, _("cannot combine if= and ifjoin="));

  if (dc3_mode_verify && dc3_vfjoin)
    dc3_error (EXIT_FAILURE, 0, _("cannot combine vf= and vfjoin="));

  if (dc3_split && (dc3_split_bytes % input_blocksize != 0))
    dc3_error (EXIT_FAILURE, 0, _("error: split size must be a multiple of block size (currently %zd)"), input_blocksize);

  if (input_file != NULL && dc3_mode_wipe)
      dc3_error (EXIT_FAILURE, 0, _("cannot combine if= and wipe="));
  // TODO: should check for wipe= and of=
  if (dc3_mode_wipe && dc3_ifjoin)
      dc3_error (EXIT_FAILURE, 0, _("cannot combine wipe= and ifjoin="));
  if (dc3_mode_wipe && dc3_vfjoin)
      dc3_error (EXIT_FAILURE, 0, _("cannot combine wipe= and vfjoin="));

  if (dc3_mode_wipe && !dc3_use_pattern)
  {
      dc3_use_pattern   = true;
      dc3_pattern_value = make_pattern("00");
      if (NULL == dc3_pattern_value)
          dc3_error(EXIT_FAILURE,0, _("illegal pattern %s"), quote("00"));
  }


  if (dc3_ifjoin)
    dc3_generate_split_filename(dc3_ifjoin_base, dc3_ifjoin_ext, dc3_ifjoin_number, &input_file);

  if (dc3_vfjoin)
  {
    dc3_generate_split_filename(dc3_vfjoin_base, dc3_vfjoin_ext, dc3_vfjoin_number, &output_file);
    dc3_mode_verify = true;
  }
}

/* Convert argv array to a single string
 */

static char*
dc3_save_cmdline(int argc, char** argv)
{
  size_t len = 1; // at least 1, for terminating NUL

  for (int i=0; i<argc; i++)
  {
    len += strlen(argv[i]);
    if (i<(argc-1))
      len += 1;
  }

  char* cmdline = NULL;
  cmdline = xmalloc(len);

  char* cmdline_orig = cmdline;

  for (int i=0; i<argc; i++)
  {
    for (size_t j=0; j<strlen(argv[i]); j++)
    {
      *cmdline = argv[i][j];
      cmdline++;
    }
    if (i<(argc-1))
    {
      *cmdline = ' ';
      cmdline++;
    }
  }

  *cmdline = '\0';

  return cmdline_orig;
}

void dc3_print_flags(FILE* stream, bool newlines)
{
#ifdef DEFAULT_BLOCKSIZE
        fprintf(stream, " DEFAULT_BLOCKSIZE=%d", DEFAULT_BLOCKSIZE);
        if (newlines) fprintf(stream, "\n");
#endif


#ifdef DEFAULT_HASH_MD5
        fprintf(stream, " DEFAULT_HASH_MD5");
        if (newlines) fprintf(stream, "\n");
#endif

#ifdef DEFAULT_HASH_SHA1
        fprintf(stream, " DEFAULT_HASH_SHA1");
        if (newlines) fprintf(stream, "\n");
#endif

#ifdef DEFAULT_HASH_SHA256
        fprintf(stream, " DEFAULT_HASH_SHA256");
        if (newlines) fprintf(stream, "\n");
#endif

#ifdef DEFAULT_HASH_512
        fprintf(stream, " DEFAULT_HASH_512");
        if (newlines) fprintf(stream, "\n");
#endif


#ifdef DEFAULT_HASHCONV_BEFORE
        fprintf(stream, " DEFAULT_HASHCONV_BEFORE");
        if (newlines) fprintf(stream, "\n");
#endif

#ifdef DEFAULT_HASHCONV_AFTER
        fprintf(stream, " DEFAULT_HASHCONV_AFTER");
        if (newlines) fprintf(stream, "\n");
#endif


#ifdef DEFAULT_PROGRESS
        fprintf(stream, " DEFAULT_PROGRESS");
        if (newlines) fprintf(stream, "\n");
#endif

#ifdef DEFAULT_PROGRESSCOUNT
        fprintf(stream, " DEFAULT_PROGRESSCOUNT=%d", DEFAULT_PROGRESSCOUNT);
        if (newlines) fprintf(stream, "\n");
#endif

#ifdef DEFAULT_IFLAG_DIRECT
        fprintf(stream, " DEFAULT_IFLAG_DIRECT");
        if (newlines) fprintf(stream, "\n");
#endif

#ifdef DEFAULT_SIZEPROBE
        fprintf(stream, " DEFAULT_SIZEPROBE");
        if (newlines) fprintf(stream, "\n");
#endif

    fprintf(stream, "\n");
}

/* Fix up translation table. */

static void
apply_translations (void)
{
  int i;

  if (conversions_mask & C_ASCII)
    translate_charset (ebcdic_to_ascii);

  if (conversions_mask & C_UCASE)
    {
      for (i = 0; i < 256; i++)
	trans_table[i] = toupper (trans_table[i]);
      translation_needed = true;
    }
  else if (conversions_mask & C_LCASE)
    {
      for (i = 0; i < 256; i++)
	trans_table[i] = tolower (trans_table[i]);
      translation_needed = true;
    }

  if (conversions_mask & C_EBCDIC)
    {
      translate_charset (ascii_to_ebcdic);
      newline_character = ascii_to_ebcdic['\n'];
      space_character = ascii_to_ebcdic[' '];
    }
  else if (conversions_mask & C_IBM)
    {
      translate_charset (ascii_to_ibm);
      newline_character = ascii_to_ibm['\n'];
      space_character = ascii_to_ibm[' '];
    }
}

/* Apply the character-set translations specified by the user
   to the NREAD bytes in BUF.  */

static void
translate_buffer (char *buf, size_t nread)
{
  char *cp;
  size_t i;

  for (i = nread, cp = buf; i; i--, cp++)
    *cp = trans_table[to_uchar (*cp)];
}

/* If true, the last char from the previous call to `swab_buffer'
   is saved in `saved_char'.  */
static bool char_is_saved = false;

/* Odd char from previous call.  */
static char saved_char;

/* Swap NREAD bytes in BUF, plus possibly an initial char from the
   previous call.  If NREAD is odd, save the last char for the
   next call.   Return the new start of the BUF buffer.  */

static char *
swab_buffer (char *buf, size_t *nread)
{
  char *bufstart = buf;
  char *cp;
  size_t i;

  /* Is a char left from last time?  */
  if (char_is_saved)
    {
      *--bufstart = saved_char;
      (*nread)++;
      char_is_saved = false;
    }

  if (*nread & 1)
    {
      /* An odd number of chars are in the buffer.  */
      saved_char = bufstart[--*nread];
      char_is_saved = true;
    }

  /* Do the byte-swapping by moving every second character two
     positions toward the end, working from the end of the buffer
     toward the beginning.  This way we only move half of the data.  */

  cp = bufstart + *nread;	/* Start one char past the last.  */
  for (i = *nread / 2; i; i--, cp -= 2)
    *cp = *(cp - 2);

  return ++bufstart;
}

/* Add OFFSET to the input offset, setting the overflow flag if
   necessary.  */

static void
advance_input_offset (uintmax_t offset)
{
  input_offset += offset;
  if (input_offset < offset)
    input_offset_overflow = true;
}

/*
static void
dc3_rewind_input_offset (uintmax_t offset)
{
  //fprintf(stderr, "dc3_rewind_input_offset called with offset=%16jx\n", offset);
  //fprintf(stderr, "dc3_rewind_input_offset input_offset was %16jx\n", input_offset);
  uintmax_t old = input_offset;
  input_offset -= offset;
  //if (input_offset > offset)
  if (input_offset > old)
    input_offset_overflow = true;
  //fprintf(stderr, "dc3_rewind_input_offset input_offset now %16jx\n", input_offset);
}
*/

/* This is a wrapper for lseek.  It detects and warns about a kernel
   bug that makes lseek a no-op for tape devices, even though the kernel
   lseek return value suggests that the function succeeded.

   The parameters are the same as those of the lseek function, but
   with the addition of FILENAME, the name of the file associated with
   descriptor FDESC.  The file name is used solely in the warning that's
   printed when the bug is detected.  Return the same value that lseek
   would have returned, but when the lseek bug is detected, return -1
   to indicate that lseek failed.

   The offending behavior has been confirmed with an Exabyte SCSI tape
   drive accessed via /dev/nst0 on both Linux-2.2.17 and Linux-2.4.16.  */

#ifdef __linux__

# include <sys/mtio.h>

# define MT_SAME_POSITION(P, Q) \
   ((P).mt_resid == (Q).mt_resid \
    && (P).mt_fileno == (Q).mt_fileno \
    && (P).mt_blkno == (Q).mt_blkno)

static off_t
skip_via_lseek (char const *filename, int fdesc, off_t offset, int whence)
{
  struct mtget s1;
  struct mtget s2;
  bool got_original_tape_position = (ioctl (fdesc, MTIOCGET, &s1) == 0);
  /* known bad device type */
  /* && s.mt_type == MT_ISSCSI2 */

  off_t new_position = lseek (fdesc, offset, whence);
  if (0 <= new_position
      && got_original_tape_position
      && ioctl (fdesc, MTIOCGET, &s2) == 0
      && MT_SAME_POSITION (s1, s2))
    {
      dc3_error (0, 0, _("warning: working around lseek kernel bug for file (%s)\n\
  of mt_type=0x%0lx -- see <sys/mtio.h> for the list of types"),
	     filename, s2.mt_type);
      errno = 0;
      new_position = -1;
    }

  return new_position;
}
#else
# define skip_via_lseek(Filename, Fd, Offset, Whence) lseek (Fd, Offset, Whence)
#endif

/* Throw away RECORDS blocks of BLOCKSIZE bytes on file descriptor FDESC,
   which is open with read permission for FILE.  Store up to BLOCKSIZE
   bytes of the data at a time in BUF, if necessary.  RECORDS must be
   nonzero.  If fdesc is STDIN_FILENO, advance the input offset.
   Return the number of records remaining, i.e., that were not skipped
   because EOF was reached.  */

static uintmax_t
skip (int fdesc, char const *file, uintmax_t records, size_t blocksize,
      char *buf)
{
  uintmax_t offset = records * blocksize;

  /* Try lseek and if an error indicates it was an inappropriate operation --
     or if the file offset is not representable as an off_t --
     fall back on using read.  */

  errno = 0;
  if (records <= OFF_T_MAX / blocksize
      && 0 <= skip_via_lseek (file, fdesc, offset, SEEK_CUR))
    {
      if (fdesc == STDIN_FILENO)
	advance_input_offset (offset);
      return 0;
    }
  else
    {
      int lseek_errno = errno;

      do
	{
	  ssize_t nread = iread (fdesc, buf, blocksize);
	  if (nread < 0)
	    {
	      if (fdesc == STDIN_FILENO)
		{
		  dc3_error (0, errno, _("skip: reading %s"), quote (file));
		  if (conversions_mask & C_NOERROR)
		    {
		      print_stats ();
		      continue;
		    }
		}
	      else
		dc3_error (0, lseek_errno, _("%s: cannot seek"), quote (file));
	      quit (EXIT_FAILURE);
	    }

	  if (nread == 0)
	    break;
	  if (fdesc == STDIN_FILENO)
	    advance_input_offset (nread);
	}
      while (--records != 0);

      return records;
    }
}

/* Advance the input by NBYTES if possible, after a read error.
   The input file offset may or may not have advanced after the failed
   read; adjust it to point just after the bad record regardless.
   Return true if successful, or if the input is already known to not
   be seekable.  */

static bool
advance_input_after_read_error (size_t nbytes)
{
  if (! input_seekable)
    {
      if (input_seek_errno == ESPIPE)
	return true;
      errno = input_seek_errno;
    }
  else
    {
      off_t offset;
      advance_input_offset (nbytes);
      input_offset_overflow |= (OFF_T_MAX < input_offset);
      if (input_offset_overflow)
	{
	  dc3_error (0, 0, _("offset overflow while reading file %s"),
		 quote (input_file));

          /* Suppress duplicate diagnostics.  */
          input_seekable = false;
          input_seek_errno = ESPIPE;

	  return false;
	}
      offset = lseek (STDIN_FILENO, 0, SEEK_CUR);
      if (0 <= offset)
	{
	  off_t diff;
	  if (offset == input_offset)
	    return true;
	  diff = input_offset - offset;
	  if (! (0 <= diff && diff <= nbytes))
          {
	    dc3_error (0, 0, _("advance: warning: invalid file offset after failed read"));
          }
	  if (0 <= skip_via_lseek (input_file, STDIN_FILENO, diff, SEEK_CUR))
	    return true;
	  if (errno == 0)
	    dc3_error (0, 0, _("cannot work around kernel bug after all"));
	}
    }

  dc3_error (0, errno, _("%s: cannot seek"), quote (input_file));

  /* Suppress duplicate diagnostics.  */
  input_seekable = false;
  input_seek_errno = ESPIPE;

  return false;
}

/*
static bool
dc3_rewind_input_after_read_error (size_t nbytes)
{
  if (nbytes == 0) // no-op
    return true;

  if (! input_seekable)
    {
      if (input_seek_errno == ESPIPE)
	return true;
      errno = input_seek_errno;
    }
  else
    {
      off_t offset;
      dc3_rewind_input_offset (nbytes);
      input_offset_overflow |= (OFF_T_MAX < input_offset);
      if (input_offset_overflow)
	{
	  dc3_error (0, 0, _("offset overflow while reading file %s"),
		 quote (input_file));
	  return false;
	}
      offset = lseek (STDIN_FILENO, 0, SEEK_CUR);
      if (0 <= offset)
	{
	  off_t diff;
	  if (offset == input_offset)
	    return true;
	  diff = input_offset - offset;
	  //if (! (0 <= diff && diff <= nbytes))
	  if (! (diff <= 0 && (-1*nbytes) <= diff) )
          {
	    dc3_error (0, 0, _("rewind: warning: invalid file offset after failed read"));
          }

	  if (0 <= skip_via_lseek (input_file, STDIN_FILENO, diff, SEEK_CUR))
	    return true;
	  if (errno == 0)
	    dc3_error (0, 0, _("cannot work around kernel bug after all"));
	}
    }

  dc3_error (0, errno, _("%s: cannot seek"), quote (input_file));
  return false;
}
*/

/* Copy NREAD bytes of BUF, with no conversions.  */

static void
copy_simple (char const *buf, size_t nread)
{
  const char *start = buf;	/* First uncopied char in BUF.  */

  do
    {
      size_t nfree = MIN (nread, output_blocksize - oc);

      memcpy (obuf + oc, start, nfree);

      nread -= nfree;		/* Update the number of bytes left to copy. */
      start += nfree;
      oc += nfree;
      if (oc >= output_blocksize)
	write_output ();
    }
  while (nread != 0);
}

/* Copy NREAD bytes of BUF, doing conv=block
   (pad newline-terminated records to `conversion_blocksize',
   replacing the newline with trailing spaces).  */

static void
copy_with_block (char const *buf, size_t nread)
{
  size_t i;

  for (i = nread; i; i--, buf++)
    {
      if (*buf == newline_character)
	{
	  if (col < conversion_blocksize)
	    {
	      size_t j;
	      for (j = col; j < conversion_blocksize; j++)
		output_char (space_character);
	    }
	  col = 0;
	}
      else
	{
	  if (col == conversion_blocksize)
	    r_truncate++;
	  else if (col < conversion_blocksize)
	    output_char (*buf);
	  col++;
	}
    }
}

/* Copy NREAD bytes of BUF, doing conv=unblock
   (replace trailing spaces in `conversion_blocksize'-sized records
   with a newline).  */

static void
copy_with_unblock (char const *buf, size_t nread)
{
  size_t i;
  char c;
  static size_t pending_spaces = 0;

  for (i = 0; i < nread; i++)
    {
      c = buf[i];

      if (col++ >= conversion_blocksize)
	{
	  col = pending_spaces = 0; /* Wipe out any pending spaces.  */
	  i--;			/* Push the char back; get it later. */
	  output_char (newline_character);
	}
      else if (c == space_character)
	pending_spaces++;
      else
	{
	  /* `c' is the character after a run of spaces that were not
	     at the end of the conversion buffer.  Output them.  */
	  while (pending_spaces)
	    {
	      output_char (space_character);
	      --pending_spaces;
	    }
	  output_char (c);
	}
    }
}

/* Set the file descriptor flags for FD that correspond to the nonzero bits
   in ADD_FLAGS.  The file's name is NAME.  */

static void
set_fd_flags (int fd, int add_flags, char const *name)
{
  /* Ignore file creation flags that are no-ops on file descriptors.  */
  add_flags &= ~ (O_NOCTTY | O_NOFOLLOW);

  if (add_flags)
    {
      int old_flags = fcntl (fd, F_GETFL);
      int new_flags = old_flags | add_flags;
      bool ok = true;
      if (old_flags < 0)
	ok = false;
      else if (old_flags != new_flags)
	{
	  if (new_flags & (O_DIRECTORY | O_NOLINKS))
	    {
	      /* NEW_FLAGS contains at least one file creation flag that
		 requires some checking of the open file descriptor.  */
	      struct stat st;
	      if (fstat (fd, &st) != 0)
		ok = false;
	      else if ((new_flags & O_DIRECTORY) && ! S_ISDIR (st.st_mode))
		{
		  errno = ENOTDIR;
		  ok = false;
		}
	      else if ((new_flags & O_NOLINKS) && 1 < st.st_nlink)
		{
		  errno = EMLINK;
		  ok = false;
		}
	      new_flags &= ~ (O_DIRECTORY | O_NOLINKS);
	    }

	  if (ok && old_flags != new_flags
	      && fcntl (fd, F_SETFL, new_flags) == -1)
	    ok = false;
	}

      if (!ok)
	dc3_error (EXIT_FAILURE, errno, _("setting flags for %s"), quote (name));
    }
}

static void
dc3_record_previous_errors(void)
{
  uintmax_t start_sec = dc3_sectors_r_full + dc3_sectors_r_partial - dc3_previous_error_count;
  uintmax_t end_sec   = dc3_sectors_r_full + dc3_sectors_r_partial - 1;

  dc3_error(0,
	    0,
	    _("Recorded %"PRIuMAX" %s from sector %"PRIuMAX" through %"PRIuMAX),
	    dc3_previous_error_count,
	    quote(strerror(dc3_previous_error_errno)),
            skip_sectors+start_sec, skip_sectors+end_sec);
}


void
dc3_read_error(void)
{
    if (dc3_group_errors)
    {
        if (!dc3_previous_error)
        {
            dc3_previous_error = true;
        }
        else if (errno != dc3_previous_error_errno)
        {
            /* We have to display the old errors before
               starting to record this new series */
            dc3_record_previous_errors();
            dc3_previous_error_count = 0;
        }

        dc3_previous_error_errno = errno;
        dc3_previous_error_count++;
    }
    else
    {
        uintmax_t block = r_full + r_partial;
        uintmax_t start_sec = block     * (input_blocksize/dc3_sectorsize);
        uintmax_t end_sec   = (block+1) * (input_blocksize/dc3_sectorsize) - 1;

        if (start_sec == end_sec)
            dc3_error( 0, errno, _("reading %s at sector %jd"), quote (input_file), skip_sectors+start_sec);
        else
            dc3_error( 0, errno, _("reading %s at sectors %jd-%jd"), quote (input_file), skip_sectors+start_sec, skip_sectors+end_sec );
    }
}

void
dc3_do_progress(void)
{
    if (    (dc3_progress_count != 0 && dc3_progress_count == dc3_progress_current)
         || 0 == dc3_progress_count )
    {
        dc3_display_progress(false);
        dc3_progress_printed = true;
        dc3_progress_current = 0;
    }
    else
        ++dc3_progress_current;
}

int dc3_recover_block(char* ibuf, ssize_t read_size, bool* eof, ssize_t* bytes_read);

/* The main loop.  */

static int
dd_copy (void)
{
  char *ibuf, *bufstart;	/* Input buffer. */
  /* These are declared static so that even though we don't free the
     buffers, valgrind will recognize that there is no "real" leak.  */
  static char *real_buf;	/* real buffer address before alignment */
  static char *real_obuf;
  ssize_t nread = 0;		/* Bytes read in the current block.  */

  /* If nonzero, then the previously read block was partial and
     PARTREAD was its size.  */
  size_t partread = 0;

  int exit_status = EXIT_SUCCESS;
  size_t n_bytes_read;

  /* Leave at least one extra byte at the beginning and end of `ibuf'
     for conv=swab, but keep the buffer address even.  But some peculiar
     device drivers work only with word-aligned buffers, so leave an
     extra two bytes.  */

  /* Some devices require alignment on a sector or page boundary
     (e.g. character disk devices).  Align the input buffer to a
     page boundary to cover all bases.  Note that due to the swab
     algorithm, we must have at least one byte in the page before
     the input buffer;  thus we allocate 2 pages of slop in the
     real buffer.  8k above the blocksize shouldn't bother anyone.

     The page alignment is necessary on any linux system that supports
     either the SGI raw I/O patch or Steven Tweedies raw I/O patch.
     It is necessary when accessing raw (i.e. character special) disk
     devices on Unixware or other SVR4-derived system.  */

  real_buf = xmalloc (input_blocksize + INPUT_BLOCK_SLOP);
  ibuf = real_buf;
  ibuf += SWAB_ALIGN_OFFSET;	/* allow space for swab */

  ibuf = ptr_align (ibuf, page_size);

  if (conversions_mask & C_TWOBUFS)
    {
      /* Page-align the output buffer, too.  */
      real_obuf = xmalloc (output_blocksize + OUTPUT_BLOCK_SLOP);
      obuf = ptr_align (real_obuf, page_size);
    }
  else
    {
      real_obuf = NULL;
      obuf = ibuf;
    }

  if (!dc3_use_pattern && skip_sectors != 0)
  {
    skip (STDIN_FILENO, input_file, skip_sectors, dc3_sectorsize, ibuf);
    /* POSIX doesn't say what to do when dd detects it has been
       asked to skip past EOF, so I assume it's non-fatal if the
       call to 'skip' returns nonzero.  FIXME: maybe give a warning.  */
  }

  if (seek_sectors != 0)
    {
      uintmax_t write_records = skip (STDOUT_FILENO, output_file,
				      seek_sectors, dc3_sectorsize, obuf);

      if (write_records != 0)
	{
	  memset (obuf, 0, output_blocksize);

	  do
	    if (iwrite(STDOUT_FILENO, obuf, output_blocksize) != output_blocksize)
	      {
		dc3_error (0, errno, _("writing to %s"), quote (output_file));
		quit (EXIT_FAILURE);
	      }
	  while (--write_records != 0);
	}
    }

  if (max_sectors == 0)
    return exit_status;

  if (dc3_use_pattern) 
    {
      size_t tmp;
      for (tmp = 0 ; tmp < input_blocksize ; ++tmp)
	ibuf[tmp] = dc3_pattern_value[tmp % dc3_pattern_len];
      nread = n_bytes_read = input_blocksize;
    }

  size_t read_size = input_blocksize;

  // TODO: this flag is kind of a lame way to keep track
  // of whether the block has been hashed yet or not, but it works.
  // now that proper tests exist, it can be refactored when time permits
  bool block_hashed = true;

  while (1)
    {
      if (dc3_sectors_r_partial + dc3_sectors_r_full >= max_sectors)
	break;


      // if reading a full block would go past EOF, reduce read_size to
      // fit remaining data exactly
      if (dc3_sectors_r_partial + dc3_sectors_r_full + input_blocksize/dc3_sectorsize >= max_sectors)
          read_size = (max_sectors - dc3_sectors_r_partial - dc3_sectors_r_full) * dc3_sectorsize;

      /* Zero the buffer before reading, so that if we get a read error,
	 whatever data we are able to read is followed by zeros.
	 This minimizes data loss. */
      if (!dc3_use_pattern)
	{
	  if ((conversions_mask & C_SYNC) && (conversions_mask & C_NOERROR))
	    memset (ibuf,
		    (conversions_mask & (C_BLOCK | C_UNBLOCK)) ? ' ' : '\0',
		    input_blocksize);
	
          //fprintf(stderr, "dd_copy: calling iread(%d, %08x, %zd), input_offset %16jx\n", STDIN_FILENO, ibuf, read_size, input_offset);  
	  nread = iread (STDIN_FILENO, ibuf, read_size);

          //if (nread != read_size)
          //  fprintf(stderr, "dd_copy: called iread(%d, %08x, %zd), input_offset %16jx, nread=%d, errno=%d\n", STDIN_FILENO, ibuf, read_size, input_offset, nread, errno);
	}
      else
        {
          nread = n_bytes_read = read_size;
        }

      bool eof = false;

      if (nread == 0)
          eof = true;

      if (nread == -1 && errno == ENOSPC)
      {
          if ( read_size == dc3_small_read || partread || !(conversions_mask&(C_SYNC|C_NOERROR)) )
              eof = true;
      }
      
      if (eof)
      {
        if ( dc3_ifjoin )
        {
            dc3_ifjoin_number++;
            dc3_generate_split_filename(dc3_ifjoin_base, dc3_ifjoin_ext, dc3_ifjoin_number, &input_file);

            if (fd_reopen (STDIN_FILENO, input_file, O_RDONLY | input_flags, 0) < 0)
            {
                //fprintf(stderr, "join mode: no more input files\n");
                dc3_ifjoin_done = true;
                break;
            }
            else
            {
                //fprintf(stderr, "\njoin: opened next file %s\n", input_file);
                // update status variables & do whatever else is needed
                // TODO: this re-probe is not the most accurate thing
                //       (would be better to stat all of the files in advance to get total expected size)
                //       this method causes the completion percentage to jump around repeatedly
                dc3_reprobe();
                continue;
            }
        }
        else
        {
	  break;
        }
      }

      block_hashed = false;

      if (nread < 0)
	{
          if (!(conversions_mask & C_DYNAMIC))
          {
            dc3_read_error();

	    if (conversions_mask & C_NOERROR)
	    {
	      if (!dc3_group_errors)
		print_stats ();

	      /* Seek past the bad block if possible. */
              if (!advance_input_after_read_error(read_size - partread))
                  exit_status = EXIT_FAILURE;

	      if ((conversions_mask & C_SYNC) && !partread)
		/* Replace the missing input with null bytes and
		   proceed normally.  */
		nread = 0;
	      else
              {
                if (dc3_hash && !block_hashed)
                  fprintf(stderr, "dd_copy: loop continue()ing without hashing block!\n");
		continue;
              }
	    }
	  else
	    {
	      /* Write any partial block. */
	      exit_status = EXIT_FAILURE;
	      break;
	    }
          }
          else
          {
            // handle error reporting when dynamic mode is applicable, but block size == sector size
            if (input_blocksize == dc3_small_read)
              dc3_read_error();

            if (!partread) /* conversions_mask && (C_DYNAMIC|C_SYNC|C_NOERROR) can be assumed true */
              nread = 0;
          }
          // else defer error handling to recovery process below
	}
      else if (dc3_group_errors && dc3_previous_error)
	/* If we've made a successful read after several errors,
	   print out the errors together now */
	{
	  dc3_record_previous_errors();
	  dc3_previous_error = false;
	  dc3_previous_error_count = 0;
	}

      n_bytes_read = nread;
      advance_input_offset (nread);
      if (dc3_hash && dc3_hash_before && !block_hashed)
      {
        //fprintf(stderr, "dd_copy() 1: calling dc3_hash_update with len=%zd\n", nread);
	dc3_hash_update(ibuf,nread);
        block_hashed = true;
      }

      //fprintf(stderr, "n_bytes_read %zd vs. read_size %zd\n", n_bytes_read, read_size);

      bool rec_eof = false;

      if (n_bytes_read < read_size)
	{
          //fprintf(stderr, "n_bytes_read < read_size)\n");
	  r_partial++;
	  partread = n_bytes_read;

	  if (conversions_mask & C_SYNC)
	    {
              // TODO: kind of a hack here to handle when we aren't really in dynamic mode
              if (input_blocksize == dc3_small_read)
              {
                /* We have to zero out the whole block now. */
	        memset (ibuf,
		        (conversions_mask & (C_BLOCK | C_UNBLOCK)) ? ' ' : '\0',
		        input_blocksize);

                /* Seek past the bad block if possible. */
                if (!advance_input_after_read_error(input_blocksize))
                  exit_status = EXIT_FAILURE;

	        n_bytes_read = read_size;
                dc3_sectors_r_partial++;
              }
              else
              {
                // TODO: replace all instances of dc3_small_read with dc3_sectorsize ?
  
                // try to read individual pieces of the block
                if ( (conversions_mask & C_DYNAMIC) && input_blocksize != dc3_small_read && (nread == -1 || nread == 0))
                {
                    int ret = dc3_recover_block(ibuf, read_size, &rec_eof, &nread);
                    if (ret != EXIT_SUCCESS)
                        exit_status = ret;
                    n_bytes_read = nread;
		    if (nread == 0)
		        break;
                }
                else
                {
                  dc3_sectors_r_full += nread / dc3_sectorsize;
                  if (nread % dc3_sectorsize != 0) dc3_sectors_r_partial++;
                }
              }
	    }
          else
            {
              dc3_sectors_r_full += n_bytes_read / dc3_sectorsize;
              if (n_bytes_read % dc3_sectorsize != 0) dc3_sectors_r_partial++;
            }
	}
      else
	{
	  r_full++;
          // TODO: make dc3_sectors_per_block variable?
          dc3_sectors_r_full += read_size / dc3_sectorsize;
	  partread = 0;
	}

      if (ibuf == obuf && !(conversions_mask&C_TWOBUFS))		/* If not C_TWOBUFS. */
	{
	  size_t nwritten = iwrite (STDOUT_FILENO, obuf, n_bytes_read);
          dc3_write_stats(nwritten);

	  if (nwritten != n_bytes_read)
	    {
	      dc3_error (0, errno, _("error writing %s"), quote (output_file));
              //fprintf(stderr, "dd_copy() loop: nwritten %d, n_bytes_read %d\n", nwritten, n_bytes_read);
	      return EXIT_FAILURE;
	    }

	  if (dc3_progress)
	   dc3_do_progress(); 

	  if (dc3_hash && !dc3_hash_before && !block_hashed)
          {
            //fprintf(stderr, "dd_copy() 2: calling dc3_hash_update with len=%zd\n", nwritten);
	    dc3_hash_update(obuf,nwritten);
            block_hashed = true;
          }

          // TODO: make this an assert
          if (dc3_hash && !block_hashed)
            fprintf(stderr, "dd_copy: loop continue()ing without hashing block!\n");

	  continue;
	}

      /* Do any translations on the whole buffer at once.  */

      if (translation_needed)
	translate_buffer (ibuf, n_bytes_read);

      if (conversions_mask & C_SWAB)
	bufstart = swab_buffer (ibuf, &n_bytes_read);
      else
	bufstart = ibuf;

      if (conversions_mask & C_BLOCK)
        copy_with_block (bufstart, n_bytes_read);
      else if (conversions_mask & C_UNBLOCK)
	copy_with_unblock (bufstart, n_bytes_read);
      else
	copy_simple (bufstart, n_bytes_read);

      if (dc3_progress)
        dc3_do_progress();
	
      
      // TODO: `n_bytes_read == blocksize' is a hack because write_output
      // can't quite handle the case where the total_bytes % blocksize != 0
      // write_output sets oc != 0 and then the last piece gets hashed twice.

      if (dc3_hash && !dc3_hash_before /*&& (n_bytes_read == input_blocksize) */ && !block_hashed)
      {
        //fprintf(stderr, "dd_copy() 3: calling dc3_hash_update with len=%zd\n", n_bytes_read);
	dc3_hash_update(bufstart,n_bytes_read);
        block_hashed = true;
      }

      if (rec_eof)
          break;

      // TODO: make this an assert
      if (dc3_hash && !block_hashed)
        fprintf(stderr, "dd_copy: loop bottom reached without hashing block!\n");
    }


  /* If we have a char left as a result of conv=swab, output it.  */
  if (char_is_saved)
    {
      if (conversions_mask & C_BLOCK)
        copy_with_block (&saved_char, 1);
      else if (conversions_mask & C_UNBLOCK)
	copy_with_unblock (&saved_char, 1);
      else
	output_char (saved_char);
    }

  if ((conversions_mask & C_BLOCK) && col > 0)
    {
      /* If the final input line didn't end with a '\n', pad
	 the output block to `conversion_blocksize' chars.  */
      size_t i;
      for (i = col; i < conversion_blocksize; i++)
	output_char (space_character);
    }

  if ((conversions_mask & C_UNBLOCK) && col == conversion_blocksize)
    /* Add a final '\n' if there are exactly `conversion_blocksize'
       characters in the final record. */
    output_char (newline_character);

  /* Write out the last block. */
  if (oc != 0)
    {
      size_t nwritten = iwrite (STDOUT_FILENO, obuf, oc);
      dc3_write_stats(nwritten);

      if (nwritten != oc)
	{
	  dc3_error (0, errno, _("dd_copy() cleanup: nwritten mismatch writing %s"), quote (output_file));
	  fprintf(stderr, "dd_copy() cleanup: nwritten %zd, n_bytes_read %zd\n", nwritten, n_bytes_read);
	  return EXIT_FAILURE;
	}
    }
  
  if (dc3_hash)
    {
      if (oc != 0 && !block_hashed)
      {
        //fprintf(stderr, "dd_copy() cleanup: calling dc3_hash_update with len=%zd\n", oc);
        //fprintf(stderr, "dd_copy() cleanup: block_hashed %d\n", block_hashed);
	dc3_hash_update(obuf,oc);
        block_hashed = true;
      }
      dc3_hash_final();
    }


  if ((conversions_mask & C_FDATASYNC) && fdatasync (STDOUT_FILENO) != 0)
    {
      if (errno != ENOSYS && errno != EINVAL)
	{
	  dc3_error (0, errno, _("fdatasync failed for %s"), quote (output_file));
	  exit_status = EXIT_FAILURE;
	}
      conversions_mask |= C_FSYNC;
    }

  if (conversions_mask & C_FSYNC)
    while (fsync (STDOUT_FILENO) != 0)
      if (errno != EINTR)
	{
	  dc3_error (0, errno, _("fsync failed for %s"), quote (output_file));
	  return EXIT_FAILURE;
	}

  if (dc3_group_errors && dc3_previous_error)
    dc3_record_previous_errors();

  // slight hack to make sure progress bar gets updated before we exit
  // because quit() doesn't update the progress bar in verify mode
  if (dc3_progress && dc3_mode_verify)
    dc3_display_progress(true);


  // TODO: make this an assert
  // TODO: fix this so it doesn't report false positives
  if (dc3_hash && !block_hashed)
    fprintf(stderr, "dd_copy: function returning without hashing block!\n");

  return exit_status;
}

int dc3_recover_block(char* ibuf, ssize_t read_size, bool* eof, ssize_t* bytes_read)
{
    int exit_status = EXIT_SUCCESS;

    // TODO: is this redundant?
    /* We have to zero out the whole block now. */
    memset (ibuf,
            (conversions_mask & (C_BLOCK | C_UNBLOCK)) ? ' ' : '\0',
            input_blocksize);

    // should already be rewound to beginning of block
    size_t i = 0;

    do
    {
        fprintf(stderr,"                                                                   \r");

        char* secbuf = ibuf+i; 

        uintmax_t sector = dc3_sectors_r_full + dc3_sectors_r_partial;

        fprintf(stderr, "trying to recover sector %llu", skip_sectors+sector);
        //fprintf(stderr, "\n");

        //fprintf(stderr, "dd_copy: recovery calling iread(%d, %08x, %zd), input_offset at %16jx\n", STDIN_FILENO, ibuf, dc3_small_read, input_offset);  
        ssize_t n = 0;
        n = iread(STDIN_FILENO, secbuf, dc3_small_read);
        //fprintf(stderr, "dd_copy: recovery called iread, nread=%d\n", n);


        if (n == 0 || (n==-1 && errno==ENOSPC)) // EOF
	{
	    *eof = true;
            //fprintf(stderr, "recovery: EOF\n");
	    break;
	}

        if (n != dc3_small_read)
        {
            //fprintf(stderr, "dd_copy recovery: read returned wrong size\n");
            // copied from above			  
            if (dc3_group_errors)
            {
                if (!dc3_previous_error)
                {
                    dc3_previous_error = true;
                }
                else if (errno != dc3_previous_error_errno)
                {
                    /* We have to display the old errors before
                       starting to record this new series */
                    fprintf(stderr,"                                                                   \r");
                    dc3_record_previous_errors();
                    dc3_previous_error_count = 0;
                }

                dc3_previous_error_errno = errno;
                dc3_previous_error_count++;
            }
            else
            {
                fprintf(stderr,"                                                                   \r");
                dc3_error (0, errno, _("reading %s at sector %jd"), quote (input_file), skip_sectors+sector);
            }

            /* Seek past the bad block if possible. */
            size_t dist = dc3_small_read;
            if (n > 0) dist -= n;
            if (!advance_input_after_read_error(dist))
                exit_status = EXIT_FAILURE;

            memset (secbuf,
                    (conversions_mask & (C_BLOCK | C_UNBLOCK)) ? ' ' : '\0',
                    dc3_small_read);

            // TODO: is this reasonable? really the whole sector failed
            // since partial sector reads aren't possible

            dc3_sectors_r_partial++;
        }
        else
        {
            //fprintf(stderr, "dd_copy recovery: read returned correct size %zd vs. %zd\n", n, dc3_small_read);

            if (dc3_group_errors && dc3_previous_error)
                /* If we've made a successful read after several errors,
                   print out the errors together now */
            {
                fprintf(stderr,"                                                                   \r");
                dc3_record_previous_errors();
                dc3_previous_error = false;
                dc3_previous_error_count = 0;
            }

            advance_input_offset (n);
            dc3_sectors_r_full++;
        }

        i+= dc3_small_read;
        fprintf(stderr,"                                                                   \r");
    } while ( i<read_size );

    *bytes_read = i;

    return exit_status;
}

int
main (int argc, char **argv)
{
  int i;
  int exit_status;
  off_t offset;

  initialize_main (&argc, &argv);
  program_name = argv[0];
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  /* Save command line string before scanargs mucks with argv. */
  char* dc3_cmdline = dc3_save_cmdline(argc, argv);

  /* Arrange to close stdout if parse_long_options exits.  */
  atexit (maybe_close_stdout);

  page_size = getpagesize ();



  if (argc == 2 && strcmp(argv[1], "--flags") == 0)
  {
    printf("%s compiled with:\n", PROGRAM_NAME);
    dc3_print_flags(stdout, true);
    exit(0);
  }


  parse_long_options (argc, argv, PROGRAM_NAME, PACKAGE, VERSION,
		      usage, AUTHORS, (char const *) NULL);
  close_stdout_required = false;

  if (getopt_long (argc, argv, "", NULL, NULL) != -1)
    usage (EXIT_FAILURE);

  /* Initialize translation table to identity translation. */
  for (i = 0; i < 256; i++)
    trans_table[i] = i;

  /* Decode arguments. */
  scanargs (argc, argv);

  apply_translations ();

  bool input_stdin = false;

  if (input_file == NULL)
    {
      if (!dc3_use_pattern)
       {
         conversions_mask |= C_TWOBUFS;
         input_file = _("standard input");
         set_fd_flags (STDIN_FILENO, input_flags, input_file);
         input_stdin = true;
       }
    }
  else
    {
#ifdef DEFAULT_IFLAG_DIRECT
      input_flags |= O_DIRECT;
#endif

      if (fd_reopen (STDIN_FILENO, input_file, O_RDONLY | input_flags, 0) < 0)
	dc3_error (EXIT_FAILURE, errno, _("opening %s"), quote (input_file));
    }

  offset = lseek (STDIN_FILENO, 0, SEEK_CUR);
  input_seekable = (0 <= offset);
  input_offset = offset;
  input_seek_errno = errno;

  // probe input device if appropriate
  if (dc3_sizeprobe && !dc3_use_pattern && !(dc3_mode_verify&&input_stdin))
  {
    if (max_sectors == (uintmax_t)-1)
      dc3_probe_file(STDIN_FILENO);
    else
      dc3_probe_sectorsize(STDIN_FILENO);
  }

  if (output_file == NULL)
    {
      output_file = _("standard output");
      set_fd_flags (STDOUT_FILENO, output_flags, output_file);
    }
  else
    {
      const char * file_to_open;

      mode_t perms = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH;
      int opts
	= (output_flags
	   | (conversions_mask & C_NOCREAT ? 0 : O_CREAT)
	   | (conversions_mask & C_EXCL ? O_EXCL : 0)
	   | (seek_sectors || (conversions_mask & C_NOTRUNC) ? 0 : O_TRUNC));

      if (dc3_split)
	{
	  /* We have to extend output file to be a new string that can hold 
	     not only the original file name but the extensions that we're
	     adding on */

	  dc3_generate_split_filename(output_file, dc3_split_format, dc3_split_number, &dc3_split_filename);

	  file_to_open = dc3_split_filename;
	}
      else
	file_to_open = output_file;

      if (dc3_pipe_output)
	{
	  close(STDOUT_FILENO);
	  FILE *tmp = popen(output_file,"w");
	  if (NULL == tmp)
	    dc3_error(EXIT_FAILURE,errno,_("opening %s"), quote(output_file));
	  int fd = fileno(tmp);
	  if (fd != STDOUT_FILENO && fd >= 0)
	    {
	      fcntl(fd,F_DUPFD,STDOUT_FILENO);
	      int saved_errno = errno;
	      close(fd);
	      errno = saved_errno;
	    }
	}


      else if (dc3_mode_verify)
	{
	  if (fd_reopen(STDOUT_FILENO,output_file,O_RDONLY,0) < 0)
	    dc3_error(EXIT_FAILURE,errno,_("opening %s"), quote(output_file));
	}


      /* Open the output file with *read* access only if we might
	 need to read to satisfy a `seek=' request.  If we can't read
	 the file, go ahead with write-only access; it might work.  */
      else if ((! seek_sectors
	   || fd_reopen (STDOUT_FILENO, file_to_open, O_RDWR | opts, perms) < 0)
	  && (fd_reopen (STDOUT_FILENO, file_to_open, O_WRONLY | opts, perms)
	      < 0))
	dc3_error (EXIT_FAILURE, errno, _("opening %s"), quote (file_to_open));

      if (dc3_sizeprobe && (dc3_use_pattern || (dc3_mode_verify&&input_stdin)))
      {
	if (max_sectors == (uintmax_t)-1)
	  dc3_probe_file(STDOUT_FILENO);
	else
	  dc3_probe_sectorsize(STDOUT_FILENO);
      }

      size_t seek_sector_size = dc3_sectorsize ? dc3_sectorsize : 512;

#if HAVE_FTRUNCATE
      if (seek_sectors != 0 && !(conversions_mask & C_NOTRUNC))
	{
          uintmax_t size = seek_sectors * seek_sector_size;
	  //unsigned long int obs = output_blocksize;

	  if (OFF_T_MAX / seek_sector_size < seek_sectors)
	    dc3_error (EXIT_FAILURE, 0,
		   _("offset too large: "
		     "cannot truncate to a length of seek=%"PRIuMAX""
		     " (%lu-byte) sectors"),
		   seek_sectors, seek_sector_size);

	  if (ftruncate (STDOUT_FILENO, size) != 0)
	    {
	      /* Complain only when ftruncate fails on a regular file, a
		 directory, or a shared memory object, as POSIX 1003.1-2004
		 specifies ftruncate's behavior only for these file types.
		 For example, do not complain when Linux 2.4 ftruncate
		 fails on /dev/fd0.  */
	      int ftruncate_errno = errno;
	      struct stat stdout_stat;
	      if (fstat (STDOUT_FILENO, &stdout_stat) != 0)
		dc3_error (EXIT_FAILURE, errno, _("cannot fstat %s"),
			   quote (file_to_open));
	      if (S_ISREG (stdout_stat.st_mode)
		  || S_ISDIR (stdout_stat.st_mode)
		  || S_TYPEISSHM (&stdout_stat))
			   dc3_error (EXIT_FAILURE, ftruncate_errno,
				      _("truncating at %"PRIuMAX" bytes in output file %s"),
				      size, quote (file_to_open));
	    }
	}
#endif
    }

  install_signal_handlers ();

  bool dc3_sizeprobe_worked = false;

  // TODO: should really specify if non-probing was due to
  // lack of size-probe mode, or probe failure
  if (dc3_sectorsize == 0)
  {
    dc3_sectorsize = 512;
    fprintf(stderr, "warning: sector size not probed, assuming 512\n");
  }
  else
  {
    dc3_sizeprobe_worked = true;
  }
  
  dc3_small_read = dc3_sectorsize;

  if (input_blocksize % dc3_sectorsize != 0)
      dc3_error(EXIT_FAILURE, 0, "fatal: input block size (%zd) must be a multiple of sector size (%zd)", input_blocksize, dc3_sectorsize);

  if (max_sectors != (uintmax_t)-1)
      input_bytes = dc3_sectorsize * max_sectors;
  // when skip=X given, adjust input_bytes so progress display is accurate
  else if (skip_sectors != 0)
      input_bytes -= skip_sectors * dc3_sectorsize;

  if (dc3_mode_wipe && max_sectors == (uintmax_t)-1 && dc3_sizeprobe_worked)
    max_sectors = input_bytes / dc3_sectorsize;

  /* We have to check if the installer turned on a hashing algorithm
     using a compiler define (e.g. CFLAGS="-DDEFAULT_HASH_MD5") but
     didn't specify a hash log or hash window. In this case the
     dc3_hash and hash_log variables don't get set. */
  i = 0;
  while (i < DC3_NUM_HASHES && !dc3_hash)
    {
      if (dc3_hashinfo[i].inuse)
	dc3_hash = true;

      i++;
    }

  if (dc3_hash)
    dc3_hash_init();

  start_time = gethrxtime ();


  char* dc3_start_time = dc3_time();
  fprintf(stderr, "dc3dd %s started at %s\n", VERSION, dc3_start_time);
  if (error_log != NULL)
    fprintf(error_log, "\ndc3dd %s started at %s\n", VERSION, dc3_start_time);

  fprintf(stderr, "command line: %s\n", dc3_cmdline);
  if (error_log != NULL)
    fprintf(error_log, "command line: %s\n", dc3_cmdline);

  fprintf(stderr, "compiled options:");
  dc3_print_flags(stderr, false);
  if (error_log != NULL)
  {
    fprintf(error_log, "compiled options:");
    dc3_print_flags(error_log, false);
  }

  fprintf(stderr, "sector size: %zd (%s)\n", dc3_sectorsize, dc3_sizeprobe_worked ? "probed" : "assumed");
  if (error_log != NULL)
    fprintf(error_log, "sector size: %zd (%s)\n", dc3_sectorsize, dc3_sizeprobe_worked ? "probed" : "assumed");

  if (error_log != NULL)
    fflush(error_log);

  exit_status = dd_copy ();

  if (dc3_mode_verify)
    dc3_error(0,0,_("Verify PASSED"));

  quit (exit_status);
}
