/*
 *  Copyright (C) 2000-2001 Marco Pesenti Gritti, Jorn Baayen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "favicon.h"
#include "mozilla.h"
#include "bookmarks.h"
#include "bookmarks_editor.h"
#include "favicon.h"
#include "eel-gconf-extensions.h"
#include "misc_general.h"
#include "misc_gui.h"
#include "embed.h"

#include <string.h>
#include <stdio.h>
#include <gtk/gtkpixmap.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

/** The favicon pixmap data cache */
static GHashTable *favicons_cache = NULL;

/** keyfile data */
static FILE *key_file            = NULL;
static FILE *host_key_file       = NULL;
static GHashTable *key_hash      = NULL;
static GHashTable *host_key_hash = NULL;

/** local functions */
static void favicon_cache (const gchar *site, const gchar *icon,
			   gboolean host_icon);
static gchar *favicon_filename_cached (const gchar *site);
static gchar *favicon_strip_cruft (const gchar *url);
static void favicon_load_hash (GHashTable *hash, const gchar *filename,
			       FILE *file);

/**
 * favicon_fetch: Parse an http URL ans get its favicon if there is one
 */
void
favicon_download (GaleonEmbed *embed, BookmarkItem *b, FaviconDownloadFlags flags)
{
	GnomeVFSURI *vfs_uri;
	gchar *favicon_url = NULL, *favicon_path, *url;
	FaviconInfo *info;
	gboolean success = FALSE;
	gboolean host_icon = FALSE;

	if (b == NULL)
		url = embed->location;
	else
		url = b->url;

	if (url == NULL) return;
	
	vfs_uri = gnome_vfs_uri_new (url);
	if (vfs_uri == NULL) return;
	
	/* we have an embed, so we get the <link rel="shortcut icon"> favicon
	 * location */
	if (embed && (flags & FAVICON_DOWNLOAD_LINK_ICON))
	{
		success = mozilla_get_favicon_location (embed, &favicon_url);
	}

	/* get host/favicon.ico if requested */
	if (!success && !favicon_url && (flags & FAVICON_DOWNLOAD_HOST_ICON))
	{
		char *uri;
		GnomeVFSURI *vfs_favicon_uri;

		uri = g_strdup (url);
		while (mozilla_uri_has_parent (uri))
		{
			char *parent_uri;
			parent_uri = mozilla_uri_get_parent (uri);
			g_free (uri);
			uri = parent_uri;
		}
		vfs_uri = gnome_vfs_uri_new (uri);
		if (!vfs_uri) return;
		vfs_favicon_uri = gnome_vfs_uri_append_file_name (vfs_uri, 
								 "favicon.ico");
		if (!vfs_favicon_uri) return;
		favicon_url = gnome_vfs_uri_to_string (vfs_favicon_uri, 0);
		gnome_vfs_uri_unref (vfs_favicon_uri);

		host_icon = TRUE;
	}
	
	gnome_vfs_uri_unref (vfs_uri);

	if (favicon_url == NULL) return;

	favicon_path = favicon_filename (favicon_url);
	if (favicon_path == NULL)
	{
		g_free (favicon_url);
		return;
	}

	/* does it exist already ? */
	if (!(flags & FAVICON_DOWNLOAD_OVERWRITE) &&
	    g_file_exists (favicon_path))
	{
		/* put in the url - favicon file */
		favicon_cache (url, favicon_path, host_icon);

		/* update the drag handke pixmap in the toolbar */
		if (embed && embed->parent_window && !b)
		{
			favicon_update_drag_handle (embed, FALSE);
		}

		/* cleanup */
		g_free (favicon_url);
		g_free (favicon_path);

		return;
	}

	/* build the download information structure */
	info = g_new0 (FaviconInfo, 1);
	info->b = b;
	info->favicon_path = favicon_path;
	info->embed = embed;
	info->url = g_strdup (url);
	info->host_icon = host_icon;

	/* remove from cache */
	if (favicons_cache != NULL)
	{
		gpointer key = NULL;
		PixmapData *val = NULL;
		
		gchar *tmp = favicon_strip_cruft (url);
		g_hash_table_lookup_extended (favicons_cache, tmp,
					      &key, (gpointer) &val);
		g_hash_table_remove (favicons_cache, tmp);
		g_free (tmp);
		
		if (val)
			misc_gui_pixmap_data_free (val);
		if (key)
			g_free (key);
	}
	
	/* commence the download */
	mozilla_save_url (embed, favicon_url, favicon_path,
			  (b != NULL ? ACTION_FAVICON_EDITOR : ACTION_FAVICON),
			  (gpointer) info);

	g_free (favicon_url);
}

/**
 * favicon_download_completed: called when the favicon download finishes,
 * does all the required actions to get the favicon into the UI
 */
void 
favicon_download_completed (FaviconInfo *info)
{
	GdkPixbuf *test;
	BookmarkItem *b = info->b;
	gchar *favicon_path;

	g_assert (info != NULL);

	favicon_path = info->favicon_path;

	g_assert (favicon_path != NULL);

	/* check whether the download succeeded */
	if (!g_file_exists (favicon_path))
	{
		g_free (favicon_path);
		g_free (info->url);
		g_free (info);
		return;
	}

	/* check whether it's a legal image */
	test = gdk_pixbuf_new_from_file (favicon_path);
	if (test == NULL)
	{
		gnome_vfs_unlink (favicon_path);
		g_free (favicon_path);
		g_free (info->url);
		g_free (info);
		return;
	}
	else
	{
		gdk_pixbuf_unref (test);
	}

	/* put in the url - favicon file */
	favicon_cache (info->url, favicon_path, info->host_icon);

	/* update the drag handke pixmap in the toolbar */
	if (info->embed && embed_exists (info->embed) &&
	    info->embed->parent_window && !b)
	{
		favicon_update_drag_handle (info->embed, FALSE);
	}
  
	/* update the bookmarks editor if we were called from there */
	if (b && bookmarks_bookmark_exists (bookmarks_root, b))
	{
		bookmarks_dirty = TRUE;
		bookmarks_toolbars_check_update (b);
		bookmarks_editor_update_tree_items (b);
	}

	g_free (favicon_path);
	g_free (info->url);
	g_free (info);
}

/**
 * favicon_filename: parse a URL and get the filename of the associated favicon
 */
gchar *
favicon_filename (gchar *favicon_url)
{
	GnomeVFSURI *vfs_uri;
	gchar *result, *slashpos, *filename, *tmp;

	vfs_uri = gnome_vfs_uri_new (favicon_url);
	if (vfs_uri == NULL)
	{
		g_free (favicon_url);
		return NULL;
	}
	
	filename = gnome_vfs_uri_to_string (vfs_uri, 
				    GNOME_VFS_URI_HIDE_USER_NAME |
				    GNOME_VFS_URI_HIDE_PASSWORD |
				    GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD |
				    GNOME_VFS_URI_HIDE_FRAGMENT_IDENTIFIER);

	while ((slashpos = strstr (filename, "/")) != NULL) *slashpos = '_';
	if (filename[strlen (filename) - 1] == '_')
		tmp = g_strndup (filename, strlen (filename) - 1);
	else
		tmp = g_strdup (filename);
	g_free (filename);

	result = g_strconcat (g_get_home_dir (), "/.galeon/favicons/",
			      tmp, NULL);
	g_free (tmp);

	gnome_vfs_uri_unref (vfs_uri);

	return result;
}

/**
 * favicon_update_drag_handle: update the drag handle pixmap in the toolbar
 */
void
favicon_update_drag_handle (GaleonEmbed *embed, gboolean fetch)
{
	const PixmapData *drag_pixmap;

	drag_pixmap = favicon_get_pixmap (embed->location);

	/* update the drag handle pixmap */
	if (embed->is_active && embed->parent_window
	    && embed->parent_window->drag_pixmap != NULL)
	{
		gtk_pixmap_set (GTK_PIXMAP
				(embed->parent_window->drag_pixmap), 
				drag_pixmap->pixmap, 
				drag_pixmap->mask);
	}

	/* update icon in notebook */
	if (embed && embed->notebook_icon)
	{
		/* handle all different configs */
		switch (eel_gconf_get_integer (CONF_TABS_TABBED_FAVICONS))
		{
		case 0:
			if (GTK_WIDGET_VISIBLE (embed->notebook_icon))
			{
				gtk_widget_hide (embed->notebook_icon);
				embed_update_tab_size (embed);
			}
			break;
		case 2:
			if (drag_pixmap == site_pixmap_data)
			{
				if (GTK_WIDGET_VISIBLE (embed->notebook_icon))
				{
					gtk_widget_hide (embed->notebook_icon);
					embed_update_tab_size (embed);
				}
				break;
			}
			/* fall through */
		default:
			if (!GTK_WIDGET_VISIBLE (embed->notebook_icon))
			{
				gtk_widget_show (embed->notebook_icon);
				embed_update_tab_size (embed);
			}
			if (drag_pixmap->pixmap ==
			    GTK_PIXMAP (embed->notebook_icon)->pixmap)
			{
				break;
			}
			gtk_pixmap_set (GTK_PIXMAP (embed->notebook_icon), 
        	        	        drag_pixmap->pixmap, drag_pixmap->mask);
			break;
		}
	}

	/* download the favicon if it's not there yet and we're asked
	 * to do so */
	if (drag_pixmap == site_pixmap_data && fetch)
	{
		if (eel_gconf_get_boolean (CONF_GENERAL_HOST_FAVICONS))
		{
			favicon_download (embed, NULL, FAVICON_DOWNLOAD_LINK_ICON |
						       FAVICON_DOWNLOAD_HOST_ICON);
		}
		else
		{
			favicon_download (embed, NULL, FAVICON_DOWNLOAD_LINK_ICON);
		}
	}
}

/**
 * favicon_get_pixmap: Parse an http URL and return the appropriate icon 
 */
const PixmapData *
favicon_get_pixmap (const gchar *url)
{
	PixmapData *result = NULL;
	gchar *favicon_path;

	if (url == NULL || strcmp (url, "") == 0 ||
	    strcmp (url, "about:blank") == 0)
	{
		return site_pixmap_data;
	}

	/* get the favicon filename */
	favicon_path = favicon_filename_cached (url);
	if (!favicon_path) return site_pixmap_data;
	
	/* check if it's cached already, if so, return the cached pixmap */
	if (favicons_cache != NULL)
	{
		result = g_hash_table_lookup (favicons_cache, favicon_path);
		/* perhaps we need to add a reference to the GdkPixmap
		   and the GdkBitmap? */
		if (result != NULL) return result;
	}
	
	/* if the favicon exists, make the pixmap and cache it */
	if (favicon_path && g_file_exists (favicon_path))
	{
		GdkPixbuf *pixbuf;
		
		pixbuf = gdk_pixbuf_new_from_file (favicon_path);
		if (pixbuf != NULL)
		{
			if (gdk_pixbuf_get_width (pixbuf) > 16 || 
			    gdk_pixbuf_get_height (pixbuf) > 16)
			{
				GdkPixbuf *scaled_icon = 
					gdk_pixbuf_scale_simple (pixbuf, 
						16, 16, GDK_INTERP_NEAREST);
				gdk_pixbuf_unref (pixbuf);
				pixbuf = scaled_icon;
			}

			/* threshold out alpha if possible */
			if (gdk_pixbuf_get_has_alpha       (pixbuf) &&
			    gdk_pixbuf_get_bits_per_sample (pixbuf) == 8 &&
			    gdk_pixbuf_get_colorspace      (pixbuf) ==
			    	GDK_COLORSPACE_RGB &&
			    gdk_pixbuf_get_n_channels      (pixbuf) == 4)
			{
				misc_gui_pixbuf_threshold_alpha (pixbuf, FALSE);
			}
			
			result = g_new (PixmapData, 1);
			gdk_pixbuf_render_pixmap_and_mask (pixbuf,
							   &(result->pixmap),
							   &(result->mask),
							   100);
			gdk_pixbuf_unref (pixbuf);
	
			if (favicons_cache == NULL) 
			{
				favicons_cache = g_hash_table_new (g_str_hash,
							 	   g_str_equal);
			}

			g_hash_table_insert (favicons_cache, favicon_path, result);
		}
		else
		{
			result = site_pixmap_data;
		}
	}
	else
	{
		result = site_pixmap_data;
	}

	return result;
}

/**
 * favicon_load_hash: init a hashtable from a keyfile
 */
static void
favicon_load_hash (GHashTable *hash, const gchar *filename, FILE *file)
{
	gchar *line = NULL;

	/* open the keyfile */
	file = fopen (filename, "r");

	/* read file */
	while (file && (line = misc_general_read_line_from_file (file)) &&
	       strlen (line) > 0)
	{
		gchar **list;

		/* initialise line */
		if (line[strlen(line) - 1] == '\n')
		{
			line[strlen(line) - 1] = ' ';
		}
		list = g_strsplit (line, " ", 0);

		/* sanity checks */
		if (list[0] != NULL && list[1] != NULL)
		{
			if (g_file_exists (list[1]))
			{
				/* put in into the hashtable */
				g_hash_table_insert (hash,
						     g_strdup (list[0]),
						     g_strdup (list[1]));
			}
		}

		/* cleanup */
		g_strfreev (list);
		g_free (line);
	}
	g_free (line);

	/* close file */
	if (file) fclose (file);
}

/**
 * favicon_key_files_init: initialise the favicon keyfiles
 */
void
favicon_key_files_init (void)
{
	gchar *filename;

	/* load the normal favicons hash */
	key_hash = g_hash_table_new (g_str_hash, g_str_equal);
	filename = g_strconcat (g_get_home_dir (), "/.galeon/favicons.keys", NULL);
	favicon_load_hash (key_hash, filename, key_file);
	key_file = fopen (filename, "a");
	if (!key_file)
		g_warning ("Error opening %s for write. "
			   "Your favicons will not be saved", filename);
	g_free (filename);

	/* load the host/favicon.ico favicons hash */
	host_key_hash = g_hash_table_new (g_str_hash, g_str_equal);
	filename = g_strconcat (g_get_home_dir (), "/.galeon/host_favicons.keys", NULL);
	favicon_load_hash (host_key_hash, filename, host_key_file);
	host_key_file = fopen (filename, "a");
	if (!host_key_file)
		g_warning ("Error opening %s for write. "
			   "Your host favicons will not be saved", filename);
	g_free (filename);
}

/**
 * favicon_key_files_close: close the keyfiles
 */
void
favicon_key_files_close (void)
{
	/* close the keyfiles */
	if (key_file)
		fclose (key_file);
	if (host_key_file)
		fclose (host_key_file);
}

/**
 * favicon_cache: cache favicon filename for a site
 */
static void
favicon_cache (const gchar *site, const gchar *icon, gboolean host_icon)
{
	gchar *tmp;
	FILE *file;
	GHashTable *hash;

	if (!g_file_exists (icon)) return;

	if (host_icon)
	{
		GnomeVFSURI *vfs_uri;
	
		vfs_uri = gnome_vfs_uri_new (site);
		if (vfs_uri == NULL) return;
		tmp = g_strdup (gnome_vfs_uri_get_host_name (vfs_uri));
		gnome_vfs_uri_unref (vfs_uri);
		if (tmp == NULL) return;

		hash = host_key_hash;
		file = host_key_file;
	}
	else
	{
		tmp = favicon_strip_cruft (site);
		hash = key_hash;
		file = key_file;
	}

	/* check it's not already there */
	if (g_hash_table_lookup (hash, tmp) == NULL)
	{
		/* put it in the keyfile */
		gchar *key = g_strdup_printf ("%s %s\n", tmp, icon); 
		fputs (key, file);
		g_free (key);
		fflush (file);

		/* put it in the hashtable */
		g_hash_table_insert (hash, tmp, g_strdup (icon));
	}
	else
	{
		g_free (tmp);
	}
}

/**
 * favicon_filename_cached: get favicon filename for a site,
 * from the key hashes
 */
static gchar *
favicon_filename_cached (const gchar *site)
{
	gchar *ret, *tmp;

	/* get a sane url */
	tmp = favicon_strip_cruft (site);
	if (!tmp) return NULL;

	/* first look for a <link> favicon */
	ret = g_hash_table_lookup (key_hash, tmp);
	g_free (tmp);

	/* no <link> favicon, look for a host/favicon.ico */
	if (ret == NULL)
	{
		GnomeVFSURI *vfs_uri;
	
		vfs_uri = gnome_vfs_uri_new (site);
		if (vfs_uri == NULL) return NULL;

		tmp = (gchar *) gnome_vfs_uri_get_host_name (vfs_uri);

		if (tmp) ret = g_hash_table_lookup (host_key_hash, tmp);

		gnome_vfs_uri_unref (vfs_uri);
	}

	/* we're done */
	return ret;
}

/**
 * favicon_strip_cruft: strip the trailing slash and hash from a filename/url
 */
static gchar *
favicon_strip_cruft (const gchar *url)
{
	GnomeVFSURI *vfs_uri;
	gchar *result, *filename;

	vfs_uri = gnome_vfs_uri_new (url);
	if (vfs_uri == NULL)
		return NULL;
	
	filename = gnome_vfs_uri_to_string (vfs_uri, 
				    GNOME_VFS_URI_HIDE_USER_NAME |
				    GNOME_VFS_URI_HIDE_PASSWORD |
				    GNOME_VFS_URI_HIDE_FRAGMENT_IDENTIFIER);

	if (filename[strlen (filename) - 1] == '/')
		result = g_strndup (filename, strlen (filename) - 1);
	else
		result = g_strdup (filename);

	g_free (filename);
	gnome_vfs_uri_unref (vfs_uri);

	return result;
}
