/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <string.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/debug.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/status-item.h>

#include <pan/filters/filter-aggregate.h>
#include <pan/filters/filter-binary.h>
#include <pan/filters/filter-cached.h>
#include <pan/filters/filter-manager.h>
#include <pan/filters/filter-new.h>
#include <pan/filters/filter-phrase.h>
#include <pan/filters/filter-read.h>
#include <pan/filters/filter-saved.h>
#include <pan/filters/filter-thread.h>
#include <pan/filters/filter-top.h>
#include <pan/filters/filter-manager.h>

#include <pan/article-list-toolbar-mediator.h> /* just for the util */
#include <pan/article-toolbar.h>
#include <pan/filter-ui.h>
#include <pan/filter-edit-ui.h>
#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/util.h>

#include <xpm/disk.xpm>
#include <xpm/filter.xpm>
#include <xpm/binary_complete.xpm>
#include <xpm/binary_incomplete.xpm>
#include <xpm/binary.xpm>
#include <xpm/bluecheck.xpm>
#include <xpm/new_unread.xpm>
#include <xpm/new_read.xpm>
#include <xpm/old_unread.xpm>
#include <xpm/watched.xpm>
#include <xpm/by_me.xpm>
#include <xpm/by_others.xpm>

static GtkWidget * show_matches_button = NULL;
static GtkWidget * show_subthread_button = NULL;
static GtkWidget * show_references_button = NULL;
static GtkWidget * show_thread_button = NULL;

static GtkWidget * show_unread_button = NULL;
static GtkWidget * show_read_button = NULL;
static GtkWidget * show_new_button = NULL;
static GtkWidget * show_old_button = NULL;
static GtkWidget * show_complete_binary_button = NULL;
static GtkWidget * show_incomplete_binary_button = NULL;
static GtkWidget * show_nonbinary_button = NULL;
static GtkWidget * show_watched_button = NULL;
static GtkWidget * show_killfile_button = NULL;
static GtkWidget * show_normal_rank_button = NULL;
static GtkWidget * show_saved_button = NULL;
static GtkWidget * show_queued_button = NULL;
static GtkWidget * show_idle_button = NULL;
static GtkWidget * show_cached_button = NULL;
static GtkWidget * show_non_cached_button = NULL;
static GtkWidget * show_by_me_button = NULL;
static GtkWidget * show_by_others_button = NULL;
static GtkWidget * _custom_filters_menu = NULL;
static GtkWidget * _no_customs_button = NULL;

static GtkWidget * _filter_phrase_entry = NULL;

static GHashTable * _name_to_menu = NULL;

static GSList * _custom_group = NULL;
static Filter * _scratch_filter = NULL;
static Filter * _filter = NULL;
static gchar * _filter_phrase_str = NULL;
static FilterPhraseState _filter_phrase_state = FILTER_PHRASE_SUBJECT;
static gint _prev_filter_phrase_state = -1;
static FilterShow _filter_show = FILTER_SHOW_MATCHES_AND_REFERENCES;

static void fire_filter_changed (void);

/*****
******  CREATE TOOLBAR
*****/

extern GtkTooltips *ttips;

static GtkWidget*
create_menu_separator (void)
{
	GtkWidget * w = gtk_menu_item_new ();
	gtk_widget_show (w);
	return w;
}

static const gchar*
get_on_the_fly_name (void)
{
	return _("On the Fly");
}


/***
****
****  STATE_FILTER ---> Filter*
**** 
***/

static gboolean
is_active (GtkWidget * w)
{
	return GTK_CHECK_MENU_ITEM(w)->active;
}

static gulong
get_filter_bits (void)
{
	gulong l = 0;
	debug_enter ("get_filter_bits");

	if (is_active (show_new_button))
		l |= STATE_FILTER_NEW;
	if (is_active (show_old_button))
		l |= STATE_FILTER_OLD;
	if (is_active (show_unread_button))
		l |= STATE_FILTER_UNREAD;
	if (is_active (show_read_button))
		l |= STATE_FILTER_READ;

	if (is_active (show_complete_binary_button))
		l |= STATE_FILTER_COMPLETE_BINARIES;
	if (is_active (show_incomplete_binary_button))
	 	l |= STATE_FILTER_INCOMPLETE_BINARIES;
	if (is_active (show_nonbinary_button))
		l |= STATE_FILTER_NONBINARIES;

	if (is_active (show_watched_button))
		l |= STATE_FILTER_WATCHED;
	if (is_active (show_killfile_button))
		l |= STATE_FILTER_IGNORED;
	if (is_active (show_normal_rank_button))
		l |= STATE_FILTER_NORMAL_RANK;

	if (is_active (show_saved_button))
		l |= STATE_FILTER_SAVED;
	if (is_active (show_idle_button))
	 	l |= STATE_FILTER_IDLE;
	if (is_active (show_queued_button))
	 	l |= STATE_FILTER_QUEUED;

	if (is_active (show_cached_button))
	 	l |= STATE_FILTER_CACHED;
	if (is_active (show_non_cached_button))
	 	l |= STATE_FILTER_NOT_CACHED;

	if (is_active (show_by_me_button))
		l |= STATE_FILTER_MINE;
	if (is_active (show_by_others_button))
		l |= STATE_FILTER_NOT_MINE;

	debug_exit ("get_filter_bits");
	return l;
}

static Filter*
create_filter_from_settings (void)
{
	return filter_create_from_bits_with_phrase (get_filter_bits(),
	                                            _filter_phrase_state,
	                                            _filter_phrase_str);
}

/***
****
****  Filter Bit Menu Buttons from STATE_FILTER
**** 
***/

static gboolean ignore_toggle = FALSE;

static void
update_filter_show_menu_buttons (FilterShow s)
{
	debug_enter ("update_filter_show_menu_buttons");

	pan_lock ();
	ignore_toggle = TRUE;

	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_matches_button),
	                                s == FILTER_SHOW_MATCHES);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_subthread_button),
	                                s == FILTER_SHOW_MATCHES_AND_REPLIES);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_references_button),
	                                s == FILTER_SHOW_MATCHES_AND_REFERENCES);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_thread_button),
	                                s == FILTER_SHOW_THREAD);

	ignore_toggle = FALSE;
	pan_unlock ();
	debug_exit ("update_filter_show_menu_buttons");
}

static void
update_filter_bit_menu_buttons (gulong l)
{
	debug_enter ("update_filter_bit_menu_buttons");

	pan_lock ();
	ignore_toggle = TRUE;

	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_new_button),
	                                l & STATE_FILTER_NEW);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_old_button),
	                                l & STATE_FILTER_OLD);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_unread_button),
	                                l & STATE_FILTER_UNREAD);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_read_button),
	                                l & STATE_FILTER_READ);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_complete_binary_button),
	                                l & STATE_FILTER_COMPLETE_BINARIES);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_incomplete_binary_button),
	                                l & STATE_FILTER_INCOMPLETE_BINARIES);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_nonbinary_button),
	                                l & STATE_FILTER_NONBINARIES);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_watched_button),
	                                l & STATE_FILTER_WATCHED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_killfile_button),
	                                l & STATE_FILTER_IGNORED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_normal_rank_button),
	                                l & STATE_FILTER_NORMAL_RANK);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_saved_button),
	                                l & STATE_FILTER_SAVED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_queued_button),
	                                l & STATE_FILTER_QUEUED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_idle_button),
	                                l & STATE_FILTER_IDLE);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_cached_button),
	                                l & STATE_FILTER_CACHED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_non_cached_button),
	                                l & STATE_FILTER_NOT_CACHED);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_by_me_button),
	                                l & STATE_FILTER_MINE);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(show_by_others_button),
	                                l & STATE_FILTER_NOT_MINE);

	ignore_toggle = FALSE;
	pan_unlock ();
	debug_exit ("update_filter_bit_menu_buttons");
}

/****
*****
*****  FILTER STATE
*****
****/

/**
 * As a point of visual feedback, we disable the bit-filter buttons
 * whenever a custom filter is active.  This function is called by
 * other gtk signal handlers that listen for either a custom filter
 * or the "turn custom filters off" button to be pressed.
 */
static void
sensitize_builtin_filters (gboolean b)
{
	debug_enter ("sensitize_builtin_filters");

	gtk_widget_set_sensitive (show_unread_button, b);
	gtk_widget_set_sensitive (show_read_button, b);
	gtk_widget_set_sensitive (show_new_button, b);
	gtk_widget_set_sensitive (show_old_button, b);

	gtk_widget_set_sensitive (show_complete_binary_button, b);
	gtk_widget_set_sensitive (show_incomplete_binary_button, b);
	gtk_widget_set_sensitive (show_nonbinary_button, b);

	gtk_widget_set_sensitive (show_watched_button, b);
	gtk_widget_set_sensitive (show_killfile_button, b);
	gtk_widget_set_sensitive (show_normal_rank_button, b);

	gtk_widget_set_sensitive (show_saved_button, b);
	gtk_widget_set_sensitive (show_queued_button, b);
	gtk_widget_set_sensitive (show_idle_button, b);

	gtk_widget_set_sensitive (show_cached_button, b);
	gtk_widget_set_sensitive (show_non_cached_button, b);

	gtk_widget_set_sensitive (show_by_me_button, b);
	gtk_widget_set_sensitive (show_by_others_button, b);

	debug_exit ("sensitize_builtin_filters");
}

static void
use_bit_filter (void)
{
	Filter * old_filter;
	debug_enter ("use_bit_filter");

	sensitize_builtin_filters (TRUE);

	/* update the filter variable */
	old_filter = _filter;
	_filter = create_filter_from_settings ();
	if (old_filter != NULL)
		pan_object_unref (PAN_OBJECT(old_filter));

	/* notify clients of the change */
	fire_filter_changed ();
	debug_enter ("use_bit_filter");
}


/****
*****
*****  FILTER POPDOWN MENU: CUSTOM FILTERS
*****
****/

static void
activate_canned_filter_by_name (const gchar * name)
{
	Filter * old_filter = _filter;
	debug_enter ("activate_canned_filter_by_name");

	/* sanity clause */
	g_return_if_fail (is_nonempty_string(name));

	/* turn off the bit-filter buttons */
	sensitize_builtin_filters (FALSE);

	/* update the _filter variable */
	if (!pan_strcmp (name, get_on_the_fly_name()))
		_filter = filter_dup (_scratch_filter);
	else
		_filter = filter_manager_get_named_filter (name);
	if (old_filter != NULL)
		pan_object_unref (PAN_OBJECT(old_filter));

	/* notify clients of the change */
	fire_filter_changed ();

	debug_exit ("activate_canned_filter_by_name");
}

static void
custom_filter_selected_cb (GtkCheckMenuItem * item, gpointer user_data)
{
	debug_enter ("custom_menu_toggle_cb");

	/*
	 * Since all custom filter activity (on/off) is now handled
	 * through a single group of radio buttons, we can ignore
	 * the events where radio buttons become inactive, as they're 
	 * always followed by a radio button becoming active.
	 */

	if (item->active == TRUE)
	{
		if (GTK_WIDGET(item) == _no_customs_button)
		{
			use_bit_filter ();
		}
		else
		{
			const gchar * name = gtk_object_get_data (GTK_OBJECT(item), "filter_name");

			if (is_nonempty_string(name))
				activate_canned_filter_by_name (name);

			/* Custom filter: erase filter phrase */
 			replace_gstr (&_filter_phrase_str, NULL);
			gtk_editable_delete_text (GTK_EDITABLE(gnome_entry_gtk_entry(GNOME_ENTRY(_filter_phrase_entry))), 0, -1);
		}
	}

	debug_exit ("custom_menu_toggle_cb");
}

static void
edit_dialog_clicked_cb (GnomeDialog * dialog, gint index, gpointer data)
{
	debug_enter ("edit_dialog_clicked_cb");

	if (index != FILTER_EDIT_DIALOG_CLICKED_CANCEL)
	{
		FilterTop * ft;

		/* update our scratch filter */
		if (1) {
			Filter * old_filter;
			GtkWidget * w;

			/* make a copy of the dialog's scratch filter */
			w = GTK_WIDGET(dialog);
			old_filter = _scratch_filter;
			_scratch_filter = filter_dup (FILTER(filter_edit_dialog_get_filter(w)));
			if (old_filter != NULL)
				pan_object_unref (PAN_OBJECT(old_filter));
		}

		/* if _filter is a copy of _scratch, then update _filter too */
		if (IS_FILTER_TOP(_filter))
		{
			ft = FILTER_TOP(_filter);
			if (!pan_strcmp(ft->name, FILTER_TOP(_scratch_filter)->name))
			{
				/* copy _scratch to _filter */
				Filter * old_filter = _filter;
				_filter = filter_dup (_scratch_filter);
				if (old_filter != NULL)
					pan_object_unref (PAN_OBJECT(old_filter));

				/* fire an update */
				fire_filter_changed ();
			}
		}
	}

	if (index != FILTER_EDIT_DIALOG_CLICKED_APPLY)
		gnome_dialog_close (dialog);

	debug_exit ("edit_dialog_clicked_cb");
}

static void
edit_on_the_fly_cb (GtkMenuItem * item, gpointer data)
{
	GtkWidget * dialog;
	debug_enter ("edit_on_the_fly_cb");

	dialog = filter_edit_dialog_new (FILTER_TOP(_scratch_filter));
	gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
	                    GTK_SIGNAL_FUNC(edit_dialog_clicked_cb), NULL);
	gtk_widget_show_all (dialog);

	debug_exit ("edit_on_the_fly_cb");
}

static void
edit_filters_cb (GtkMenuItem * item, gpointer data)
{
	GtkWidget * w = filter_dialog_new ();
	gnome_dialog_set_parent (GNOME_DIALOG(w), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (w);
}


static void
update_custom_menus (void)
{
	guint i;
	gchar * pch;
	GPtrArray * filters;
	GtkMenu * menu;
	GtkWidget * w;
	debug_enter ("update_custom_menus");

	g_return_if_fail (_custom_filters_menu!=NULL);

	menu = GTK_MENU(gtk_menu_new());

	/* get the filters */
	filters = g_ptr_array_new ();
	filter_manager_get_filters (filters);
	g_ptr_array_add (filters, filter_dup(_scratch_filter));

	/* create a hashtable for finding the menu button */
	if (_name_to_menu != NULL) {
		/* FIXME: g_free the key strings */
		g_hash_table_destroy (_name_to_menu);
	}
	_name_to_menu = g_hash_table_new (g_str_hash, g_str_equal);

	/* add the custom filters */
	_custom_group = NULL;

	/* add the "no custom filter" radio button */
	_no_customs_button = w = gtk_radio_menu_item_new_with_label (_custom_group, _("No Custom Filter"));
	_custom_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM(w));
	gtk_signal_connect (GTK_OBJECT(w), "toggled", custom_filter_selected_cb, NULL);
	gtk_widget_show (w);
	gtk_menu_append (menu, w);

	/* add separator */
	gtk_menu_append (menu, create_menu_separator());

	for (i=0; i<filters->len; ++i)
	{
		FilterTop * ft = FILTER_TOP (g_ptr_array_index(filters,i));
		if (ft==NULL || !ft->is_visible)
			continue;

		w = gtk_radio_menu_item_new_with_label (_custom_group, ft->name);
		_custom_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM(w));
		gtk_object_set_data_full (GTK_OBJECT(w), "filter_name", g_strdup(ft->name), g_free);
		gtk_signal_connect (GTK_OBJECT(w), "toggled", custom_filter_selected_cb, NULL);
		gtk_widget_show (w);
		gtk_menu_append (menu, w);

		g_hash_table_insert (_name_to_menu, g_strdup(ft->name), w);
	}

	/* add separator */
	gtk_menu_append (menu, create_menu_separator());

	/* add "edit on-the-fly" */
	pch = g_strdup_printf (_("Edit \"%s\" Filter"), get_on_the_fly_name());
	w = gtk_menu_item_new_with_label (pch);
	g_free (pch);
	gtk_signal_connect (GTK_OBJECT(w), "activate", edit_on_the_fly_cb, NULL);
	gtk_widget_show (w);
	gtk_menu_append (menu, w);

	/* add "edit filters" */
	w = gtk_menu_item_new_with_label (_("Edit Other Filters..."));
	gtk_signal_connect (GTK_OBJECT(w), "activate", edit_filters_cb, NULL);
	gtk_widget_show (w);
	gtk_menu_append (menu, w);

	/* install the menu */
	gtk_widget_show (GTK_WIDGET(menu));
	gtk_menu_item_set_submenu (GTK_MENU_ITEM(_custom_filters_menu), GTK_WIDGET(menu));

	/* cleanup */
	pan_g_ptr_array_foreach (filters, (GFunc)pan_object_unref, NULL);
	g_ptr_array_free (filters, TRUE);
	debug_exit ("update_custom_menus");
}


/****
*****
*****  FILTER POPDOWN MENU: FILTER BITS
*****
****/

static void
filter_bit_toggled_cb (GtkCheckMenuItem * tb, gpointer user_data)
{
	debug_enter ("filter_bit_toggled_cb");

	if (!ignore_toggle)
	{
		/* update filter */
		Filter * old_filter = _filter;
		_filter = create_filter_from_settings ();
		if (old_filter != NULL) {
			pan_object_unref (PAN_OBJECT(old_filter));
		}

		/* update to listeners that filter has changed */
		fire_filter_changed ();
	}

	debug_exit ("filter_bit_toggled_cb");
}

static GtkWidget*
bit_menu_new (const gchar * label, gchar ** xpm, const gchar * stock, GtkWidget ** setme)
{
	GtkWidget * l;
	GtkWidget * p;
	GtkWidget * w;
	GtkWidget * box;
	debug_enter ("bit_menu_new");

	box = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);

	if (xpm != NULL)
		p = gnome_pixmap_new_from_xpm_d (xpm);
	else if (is_nonempty_string (stock))
		p = gnome_stock_pixmap_widget_new (Pan.window, stock);
	else
		p = gnome_stock_pixmap_widget_new (Pan.window, GNOME_STOCK_MENU_BLANK);

	l = gtk_label_new (label);
	gtk_box_pack_start (GTK_BOX(box), p, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(box), l, FALSE, FALSE, 0);

	w = gtk_check_menu_item_new ();
	gtk_container_add (GTK_CONTAINER(w), box);
	gtk_signal_connect (GTK_OBJECT(w), "toggled", filter_bit_toggled_cb, NULL);
	gtk_widget_show_all (w);
	*setme = w;
	debug_exit ("bit_menu_new");
	return w;
}

static void
custom_filters_changed_cb (gpointer foo, gpointer bar, gpointer mum)
{
	update_custom_menus ();
}

static void
match_toggled_cb (GtkCheckMenuItem * tb, gpointer user_data)
{
	debug_enter ("match_toggled_cb");

	if (!ignore_toggle)
	{
		_filter_show = (FilterShow) GPOINTER_TO_INT(user_data);
		fire_filter_changed ();
	}

	debug_exit ("match_toggled_cb");
}

static void
menu_destroyed_cb (GtkObject * menu, gpointer unused)
{
	pan_callback_remove (filter_manager_get_filters_changed_callback(),
	                     custom_filters_changed_cb,
	                     NULL);
}

static GtkWidget*
build_popdown_filter_menu (void)
{
	const gchar * cpch;
	GtkWidget * w;
	GtkMenu * m;
	GtkMenu * show;
	GSList * group;
	gpointer data;

	/* create the menu */
	m = GTK_MENU(gtk_menu_new());

	/**
	***  What to show when there's a match 
	**/

	w = gtk_menu_item_new_with_label (_("Which Articles to Show"));
	gtk_widget_show (GTK_WIDGET(w));
	gtk_menu_append (m, GTK_WIDGET(w));
	show = GTK_MENU(gtk_menu_new());
	gtk_menu_item_set_submenu (GTK_MENU_ITEM(w), GTK_WIDGET(show));

	group = NULL;
	data = GUINT_TO_POINTER(FILTER_SHOW_MATCHES);
	cpch = _("Show Matching Articles");
	w = gtk_radio_menu_item_new_with_label (group, cpch);
	group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM(w));
	gtk_signal_connect (GTK_OBJECT(w), "toggled", match_toggled_cb, data);
	gtk_widget_show_all (w);
	gtk_menu_append (show, w);
	show_matches_button = w;

	cpch = _("Show Matching Articles and Replies");
	data = GUINT_TO_POINTER(FILTER_SHOW_MATCHES_AND_REPLIES);
	w = gtk_radio_menu_item_new_with_label (group, cpch);
	group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM(w));
	gtk_signal_connect (GTK_OBJECT(w), "toggled", match_toggled_cb, data);
	gtk_widget_show_all (w);
	gtk_menu_append (show, w);
	show_subthread_button = w;

	cpch = _("Show Matching Articles and References");
	data = GUINT_TO_POINTER(FILTER_SHOW_MATCHES_AND_REFERENCES);
	w = gtk_radio_menu_item_new_with_label (group, cpch);
	group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM(w));
	gtk_signal_connect (GTK_OBJECT(w), "toggled", match_toggled_cb, data);
	gtk_widget_show_all (w);
	gtk_menu_append (show, w);
	show_references_button = w;

	cpch = _("Show Threads with Matching Articles");
	data = GUINT_TO_POINTER(FILTER_SHOW_THREAD);
	w = gtk_radio_menu_item_new_with_label (group, cpch);
	group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM(w));
	gtk_signal_connect (GTK_OBJECT(w), "toggled", match_toggled_cb, data);
	gtk_widget_show_all (w);
	gtk_menu_append (show, w);
	show_thread_button = w;
	gtk_menu_append (m, create_menu_separator());


	/* user-defined filters */
	_custom_filters_menu = gtk_menu_item_new_with_label (_("Custom Filters"));
	gtk_menu_append (m, _custom_filters_menu);
	gtk_menu_append (m, create_menu_separator());

	/* built-in filters */
	gtk_menu_append (m, bit_menu_new(_("Show New Articles"),
	                                 new_unread_xpm, NULL,
	                                 &show_new_button));
	gtk_menu_append (m, bit_menu_new(_("Show Old Articles"),
	                                 old_unread_xpm, NULL,
	                                 &show_old_button));
	gtk_menu_append (m, create_menu_separator());
	gtk_menu_append (m, bit_menu_new(_("Show Unread Articles"),
	                                 new_unread_xpm, NULL,
	                                 &show_unread_button));
	gtk_menu_append (m, bit_menu_new(_("Show Read Articles"),
	                                 new_read_xpm, NULL,
	                                 &show_read_button));
	gtk_menu_append (m, create_menu_separator());
	gtk_menu_append (m, bit_menu_new(_("Show Complete Binary Articles"),
	                                 binary_complete_xpm, NULL,
	                                 &show_complete_binary_button));
	gtk_menu_append (m, bit_menu_new(_("Show Incomplete Binary Articles"),
	                                 binary_incomplete_xpm, NULL, 
	                                 &show_incomplete_binary_button));
	gtk_menu_append (m, bit_menu_new(_("Show Text Articles"),
	                                 NULL, NULL,
	                                 &show_nonbinary_button));
	gtk_menu_append (m, create_menu_separator());
	gtk_menu_append (m, bit_menu_new(_("Show Saved Articles"),
	                                 binary_xpm, NULL,
	                                 &show_saved_button));
	gtk_menu_append (m, bit_menu_new(_("Show Queued Articles"),
	                                 bluecheck_xpm, NULL,
	                                 &show_queued_button));
	gtk_menu_append (m, bit_menu_new(_("Show Articles Neither Queued nor Saved"),
                                         NULL, NULL,
                                         &show_idle_button));
	gtk_menu_append (m, create_menu_separator());
	gtk_menu_append (m, bit_menu_new(_("Show Articles which are Cached Locally"),
	                                 disk_xpm, NULL,
                                         &show_cached_button));
	gtk_menu_append (m, bit_menu_new(_("Show Articles which are not Cached Locally"),
	                                 NULL, NULL,
	                                 &show_non_cached_button));
	gtk_menu_append (m, create_menu_separator());
	gtk_menu_append (m, bit_menu_new(_("Show Articles posted by Me"),
	                                 by_me_xpm, NULL,
                                         &show_by_me_button));
	gtk_menu_append (m, bit_menu_new(_("Show Articles posted by Others"),
	                                 by_others_xpm, NULL,
	                                 &show_by_others_button));
	gtk_menu_append (m, create_menu_separator());
	gtk_menu_append (m, bit_menu_new(_("Show Watched Threads"),
	                                 watched_xpm, NULL,
	                                 &show_watched_button));
	gtk_menu_append (m, bit_menu_new(_("Show Ignored Threads"),
	                                 NULL, GNOME_STOCK_MENU_TRASH_FULL,
	                                 &show_killfile_button));
	gtk_menu_append (m, bit_menu_new(_("Show Threads Neither Watched nor Ignored"),
	                                 NULL, NULL,
	                                 &show_normal_rank_button));
	
	/* add the submenu onto a dropdown button */
	w = gtk_menu_item_new ();
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (filter_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Filters"), "");
	gtk_menu_item_set_submenu (GTK_MENU_ITEM(w), GTK_WIDGET(m));


	/* add the custom filters cascade, and register to listen for changes
	   to the custom filters to know when the cascade needs rebuilding */
	update_custom_menus ();
	pan_callback_add (filter_manager_get_filters_changed_callback(),
	                  custom_filters_changed_cb,
	                  NULL);
	gtk_signal_connect (GTK_OBJECT(w), "destroy",
	                    GTK_SIGNAL_FUNC(menu_destroyed_cb), NULL);

	return w;
}



/****
*****
*****  TOOLBAR CALLBACKS
*****
****/

static gboolean
article_phrase_activate_cb (void)
{
	GtkWidget * e;
	gchar * s;
	debug_enter ("article_phrase_activate_cb");

	/* if the new string differs from the old one, then update. */
	e = gnome_entry_gtk_entry (GNOME_ENTRY(_filter_phrase_entry));
	s = gtk_editable_get_chars (GTK_EDITABLE(e), 0, -1);
	s = g_strstrip (s);

	if (!pan_strcmp (_filter_phrase_str, s) && _prev_filter_phrase_state==(gint)_filter_phrase_state) 
	{
		g_free (s);
	}
	else
	{
		replace_gstr (&_filter_phrase_str, s);
		_prev_filter_phrase_state = (int) _filter_phrase_state;

		if (is_active(_no_customs_button))
			use_bit_filter ();
		else
			gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM(_no_customs_button), TRUE);
	}

	debug_exit ("article_phrase_activate_cb");
	return FALSE;
}

static void
set_filter_mode_cb (GtkMenuItem * item, gpointer data)
{
	_filter_phrase_state = GPOINTER_TO_INT(data);

	article_phrase_activate_cb ();
}

/****
*****
*****  BUILDING THE TOOLBAR
*****
****/

typedef struct
{
	const gchar * name;
	FilterPhraseState state;
}
ArticleToolbarOptionMenuStruct;

GtkWidget*
article_toolbar_new (void)
{
	GtkWidget * w;
	GtkWidget * toolbar;

	_scratch_filter = filter_top_new ();
	filter_top_set_name (FILTER_TOP(_scratch_filter), get_on_the_fly_name());

	toolbar = gtk_hbox_new (FALSE, GNOME_PAD);

	/* popdown filter menu for twiddling bits */
	w = gtk_menu_bar_new ();
	gtk_menu_bar_append (GTK_MENU_BAR(w), build_popdown_filter_menu());
	gtk_box_pack_start (GTK_BOX(toolbar), w, FALSE, FALSE, 0);

	/* filter by phrase */
	if (1) {
		GtkWidget * option_menu = gtk_option_menu_new ();
		GtkWidget *menu = gtk_menu_new ();
		int index = 0;
		int i;
		ArticleToolbarOptionMenuStruct foo[] = {
			{NULL, FILTER_PHRASE_SUBJECT},
			{NULL, FILTER_PHRASE_AUTHOR},
			{NULL, FILTER_PHRASE_MESSAGE_ID}
		};
		const int row_qty = sizeof(foo) / sizeof(foo[0]);
		foo[0].name = _("Subject");
		foo[1].name = _("Author");
		foo[2].name = _("Message-ID");
		for (i=0; i<row_qty; ++i) {
			GtkWidget * item = gtk_menu_item_new_with_label (foo[i].name);
			gtk_signal_connect (GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(set_filter_mode_cb),
				GINT_TO_POINTER(foo[i].state));
			gtk_menu_append (GTK_MENU (menu), item);
			gtk_widget_show (item);
		}
		gtk_menu_set_active (GTK_MENU(menu), index);
		gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
		gtk_widget_show_all (GTK_WIDGET(option_menu));
		w = option_menu;
	}
	gtk_box_pack_start (GTK_BOX(toolbar), w, FALSE, FALSE, 0);
	_filter_phrase_entry = w = gnome_entry_new ("Article Toolbar Filter");
	gnome_entry_set_max_saved (GNOME_ENTRY(w), 10);
        gtk_signal_connect (GTK_OBJECT (gnome_entry_gtk_entry(GNOME_ENTRY(w))), "focus_out_event",
	                    GTK_SIGNAL_FUNC(article_phrase_activate_cb), NULL);
	gtk_signal_connect (GTK_OBJECT (gnome_entry_gtk_entry(GNOME_ENTRY(w))), "activate",
	                    GTK_SIGNAL_FUNC(article_phrase_activate_cb), NULL);
	gtk_signal_connect (GTK_OBJECT(GTK_COMBO(w)->popwin), "button_press_event",
	                    GTK_SIGNAL_FUNC(article_phrase_activate_cb), NULL);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
	        _("Type in a search string and press ENTER. "
	          "Wildcards are allowed; see http://pan.rebelbase.com/wildmat.html "
	          "for more information."), "");
	gtk_box_pack_start (GTK_BOX(toolbar), w, TRUE, TRUE, 0);

	/* return the toolbar widget */
	gtk_widget_show_all (toolbar);
	return toolbar;
}

/****
*****
*****  FILTER ACCESSOR/MUTATOR
*****
****/


void
article_toolbar_get_filter (Filter     ** newme_filter,
                            gulong      * bits_setme,
                            FilterShow  * show_setme)
{
	g_return_if_fail (newme_filter!=NULL);
	g_return_if_fail (bits_setme!=NULL);
	g_return_if_fail (show_setme!=NULL);

	*newme_filter = filter_dup (_filter);
	*bits_setme = get_filter_bits ();
	*show_setme = _filter_show;
}

void
article_toolbar_set_filter (const gchar  * name,
                            gulong         bits,
                            FilterShow     show)
{
	GtkWidget * w = NULL;

	debug_enter ("article_toolbar_set_filter");

	_filter_show = show;
	update_filter_show_menu_buttons (_filter_show);
	update_filter_bit_menu_buttons (bits);

	if (is_nonempty_string (name))
		w = g_hash_table_lookup (_name_to_menu, name);

	pan_lock ();
	if (w != NULL) /* we have that custom filter */
	{
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(w), TRUE);
	}
	else /* make a bit-filter */
	{
		GtkWidget * e;
		Filter * old_filter = _filter;

		_filter = create_filter_from_settings ();
		if (old_filter != NULL)
			pan_object_unref (PAN_OBJECT(old_filter));

 		replace_gstr (&_filter_phrase_str, NULL);
		e = gnome_entry_gtk_entry (GNOME_ENTRY(_filter_phrase_entry));
		gtk_editable_delete_text (GTK_EDITABLE(e), 0, -1);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(_no_customs_button), TRUE);
	}
	pan_unlock ();

	/* update to listeners that filter has changed */
	fire_filter_changed ();

	debug_exit ("article_toolbar_set_filter");
}

/***
****  Events
***/

PanCallback*
article_toolbar_get_filter_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_filter_changed (void)
{
	ArticleToolbarCallbackStruct cbs;
	debug_enter ("fire_filter_changed");

	cbs.filter = _filter;
	cbs.filter_bits = get_filter_bits ();
	cbs.filter_show = _filter_show;
	pan_callback_call (article_toolbar_get_filter_changed_callback(), &cbs, NULL);

	debug_exit ("fire_filter_changed");
}
