#include "board.h"
#include "board.moc"

#include "common/misc_ui.h"
#include "piece.h"


KSBoard::KSBoard(bool graphic, QWidget *parent)
: Board(graphic, new GiftPool(parent), parent),
  filled(matrix().height()), linesRemoved(4)
{}

void KSBoard::copy(const GenericTetris &g)
{
	Board::copy(g);
	filled = static_cast<const KSBoard &>(g).filled.copy(); // deep copy
}

void KSBoard::init(const GTInitData &data)
{
	Board::init(data);
	addRemoved = 0;
    for (uint i=0; i<4; i++) {
        linesRemoved[i] = 0;
        emit removedLinesUpdated(i, 0);
    }
}

void KSBoard::computeInfos()
{
    Board::computeInfos();

    filled.fill(0);
    for (uint j=0; j<firstClearLine(); j++)
        for (uint i=0; i<matrix().width(); i++) {
            Coord c(i, j);
            if ( matrix()[c] ) filled[j]++;
        }
}

bool KSBoard::needRemoving()
{
	for(uint j=0; j<firstClearLine(); j++)
		if ( filled[j]==matrix().width() ) return true;
	return false;
}

bool KSBoard::toBeRemoved(const Coord &c) const
{
    return ( filled[c.j]==matrix().width() );
}

void KSBoard::remove()
{
    uint nbFullLines = 0;
	for (uint k=0; k<firstClearLine(); k++)
        if ( filled[k]==matrix().width() ) nbFullLines++;

    Board::remove();

	updateRemoved(nbRemoved() + nbFullLines);
    linesRemoved[nbFullLines-1]++;
    emit removedLinesUpdated(nbFullLines-1, linesRemoved[nbFullLines-1]);
	addRemoved += nbFullLines;

	// Assign score according to level and nb of full lines (gameboy style)
	switch (nbFullLines) {
	 case 0: break;
	 case 1: updateScore( score() + 40   * level() ); break;
	 case 2: updateScore( score() + 100  * level() ); break;
	 case 3: updateScore( score() + 300  * level() ); break;
	 case 4: updateScore( score() + 1200 * level() ); break;
	}

	// increase level every 10 lines
    if ( nbRemoved()>=level()*10 ) updateLevel(level()+1);
}

bool KSBoard::toFall(const Coord &c) const
{
    return ( filled[c.j-1]==0 );
}

/*****************************************************************************/
// Multiplayers methods
uint KSBoard::gift()
{
	uint g = (addRemoved>1 ? addRemoved-1 : 0);
	addRemoved = 0;
	return g;
}

bool KSBoard::_putGift(uint nb)
{
	if ( nbClearLines()==0 ) return false;

	// lift all the cases of one line
    // (nbClearLine!=0 --> firstClearLine!=height
	for (uint j=firstClearLine(); j>0; j--)
        for (uint i=0; i<matrix().width(); i++)	{
            Coord src(i, j-1);
            Coord dest(i, j);
            moveBlock(src, dest);
        }

	// fill the emptied low line with garbage :)
	uint _nb = nb;
	uint i;
	do {
		i = randomGarbage.getLong(matrix().width());
        Coord c(i, 0);
		if ( matrix()[c]!=0 ) continue;
		Block *gb = currentPiece()->garbageBlock();
		gb->sprite()->show();
		setBlock(c, gb);
		_nb--;
	} while ( _nb!=0 );

	return true;
}

bool KSBoard::putGift(uint nb)
{
	for (uint k=0; k<nb; k++) {
		if ( !_putGift(matrix().width() - k - 1) ) return false;
        computeInfos();
    }
	return true;
}

