/*
 * kcmlisa.cpp
 *
 * Copyright (c) 2000,2001 Alexander Neundorf <neundorf@kde.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "kcmlisa.h"

#include "findnic.h"
#include "setupwizard.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <time.h>

#include <qtooltip.h>
#include <qfile.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qgrid.h>

#include <kapplication.h>
#include <kprocess.h>
#include <klocale.h>
#include <kmessagebox.h>

#include <kdebug.h>

LisaSettings::LisaSettings(const QString& config, QWidget *parent)
:QVBox(parent)
,m_config(config,false,true)
{
   m_configFilename=config;
   QHBox *hbox1 = new QHBox(this);
   new QWidget(hbox1);
   m_autoSetup=new QPushButton(i18n("&Guided LISa Setup..."),hbox1);
   m_autoSetup->setFixedWidth( m_autoSetup->sizeHint().width() );   

   QGrid *addressesGrid = new QGrid(2, Qt::Horizontal, this);
   addressesGrid->setSpacing(10);
   setStretchFactor(addressesGrid,0);

   QLabel *label=new QLabel(i18n("&Scan these addresses"),addressesGrid);
   QString comment = i18n("Enter all ranges to scan, using the format '192.168.0.1/255.255.255.255.0;10.0.0.1;255.0.0.0'");
   QToolTip::add(label,comment);
   m_pingAddresses=new KRestrictedLine(addressesGrid,"a","0123456789.-/;");
   QToolTip::add(m_pingAddresses, comment);
   label->setBuddy(m_pingAddresses);

   label=new QLabel(i18n("&Trusted addresses"),addressesGrid);
   comment = i18n("usually your network address/subnet mask (e.g. 192.168.0.0/255.255.255.0;)");
   QToolTip::add(label, comment);

   m_allowedAddresses=new KRestrictedLine(addressesGrid,"a","0123456789./;");
   QToolTip::add(m_allowedAddresses, comment);
   label->setBuddy(m_allowedAddresses);

   label=new QLabel(i18n("&Broadcast network address"),addressesGrid);
   comment=i18n("Your network address/subnet mask (e.g. 192.168.0.0/255.255.255.0;)");
   QToolTip::add(label, comment);

   m_broadcastNetwork=new KRestrictedLine(addressesGrid,"a","0123456789./;");
   QToolTip::add(m_broadcastNetwork,comment);
   label->setBuddy(m_broadcastNetwork);

   m_pingNames=new KEditListBox(i18n("A&dditionally check the following hosts"),this,"a",false, KEditListBox::Add|KEditListBox::Remove);
   QToolTip::add(m_pingNames,i18n("The names of the hosts you want to check"));

   QGroupBox *gb=new QGroupBox(i18n("Advanced settings"),this);
   gb->setColumnLayout(0, Qt::Vertical );
   gb->layout()->setSpacing( 6 );
   gb->layout()->setMargin( 6 );
   QGridLayout *grid2 = new QGridLayout( gb->layout(), 4, 4);
   m_useNmblookup=new QCheckBox(i18n("Send NetBIOS broadcasts using &nmblookup for searching"), gb);
   QToolTip::add(m_useNmblookup,i18n("only hosts running smb servers will answer"));
   grid2->addMultiCellWidget(m_useNmblookup, 0, 0, 0, 3);

   m_deliverUnnamedHosts=new QCheckBox(i18n("Re&port unnamed hosts"),gb);
   QToolTip::add(m_deliverUnnamedHosts,i18n("report hosts without DNS names"));
   grid2->addMultiCellWidget(m_deliverUnnamedHosts, 1, 1, 0, 1);

   m_secondScan=new QCheckBox(i18n("Al&ways scan twice"),gb);
   QToolTip::add(m_secondScan,i18n("check the hosts twice"));
   grid2->addMultiCellWidget(m_secondScan, 1, 1, 2, 3);

   label=new QLabel(i18n("Update interval"),gb);
   QToolTip::add(label,i18n("search hosts after this number of seconds"));
   grid2->addWidget(label, 2, 0);
   m_updatePeriod=new QSpinBox(30,1800,10,gb);
   m_updatePeriod->setSuffix(i18n(" sec"));
   QToolTip::add(m_updatePeriod,i18n("search hosts after this number of seconds"));
   grid2->addWidget(m_updatePeriod, 2, 1);

   label=new QLabel(i18n("Wait for replies after first scan"),gb);
   QToolTip::add(label,i18n("how long to wait for replies to the ICMP echo requests from hosts"));
   grid2->addWidget(label, 2, 2);
   m_firstWait=new QSpinBox(10,1000,50,gb);
   m_firstWait->setSuffix(i18n(" ms"));
   QToolTip::add(m_firstWait,i18n("how long to wait for replies to the ICMP echo requests from hosts"));
   grid2->addWidget(m_firstWait, 2, 3);

   label=new QLabel(i18n("Send pings at once"),gb);
   QToolTip::add(label,i18n("send this number of ping packets at once"));
   grid2->addWidget(label, 3, 0);
   m_maxPingsAtOnce=new QSpinBox(8,1024,5,gb);
   QToolTip::add(m_maxPingsAtOnce,i18n("send this number of ping packets at once"));
   grid2->addWidget(m_maxPingsAtOnce, 3, 1);

   label=new QLabel(i18n("Wait for replies after second scan"),gb);
   QToolTip::add(label,i18n("how long to wait for replies to the ICMP echo requests from hosts"));
   grid2->addWidget(label, 3, 2);
   m_secondWait=new QSpinBox(0,1000,50,gb);
   m_secondWait->setSuffix(i18n(" ms"));
   QToolTip::add(m_secondWait,i18n("how long to wait for replies to the ICMP echo requests from hosts"));
   grid2->addWidget(m_secondWait, 3, 3);

   connect(m_secondScan,SIGNAL(toggled(bool)),m_secondWait,SLOT(setEnabled(bool)));

   connect(m_pingAddresses,SIGNAL(textChanged(const QString&)),this,SIGNAL(changed()));
   connect(m_allowedAddresses,SIGNAL(textChanged(const QString&)),this,SIGNAL(changed()));
   connect(m_broadcastNetwork,SIGNAL(textChanged(const QString&)),this,SIGNAL(changed()));

   connect(m_pingAddresses,SIGNAL(returnPressed()),this,SIGNAL(changed()));
   connect(m_allowedAddresses,SIGNAL(returnPressed()),this,SIGNAL(changed()));
   connect(m_broadcastNetwork,SIGNAL(returnPressed()),this,SIGNAL(changed()));

   connect(m_firstWait,SIGNAL(valueChanged(int)),this,SIGNAL(changed()));
   connect(m_secondWait,SIGNAL(valueChanged(int)),this,SIGNAL(changed()));
   connect(m_maxPingsAtOnce,SIGNAL(valueChanged(int)),this,SIGNAL(changed()));
   connect(m_secondScan,SIGNAL(toggled(bool)),this,SIGNAL(changed()));
   connect(m_deliverUnnamedHosts,SIGNAL(toggled(bool)),this,SIGNAL(changed()));
   connect(m_updatePeriod,SIGNAL(valueChanged(int)),this,SIGNAL(changed()));
   connect(m_pingNames,SIGNAL(changed()),this,SIGNAL(changed()));
   connect(m_useNmblookup,SIGNAL(toggled(bool)),this,SIGNAL(changed()));
   connect(m_autoSetup,SIGNAL(clicked()),this,SLOT(autoSetup()));

   setMargin(10);
   setSpacing(15);
}

void LisaSettings::load()
{
   int secondWait=m_config.readNumEntry("SecondWait",-1);
   if (secondWait<0)
   {
      m_secondWait->setValue(300);
      m_secondScan->setChecked(FALSE);
      m_secondWait->setEnabled(FALSE);
   }
   else
   {
      m_secondWait->setValue(secondWait*10);
      m_secondScan->setChecked(TRUE);
      m_secondWait->setEnabled(TRUE);
   };
   m_deliverUnnamedHosts->setChecked(m_config.readNumEntry("DeliverUnnamedHosts",0));

   m_firstWait->setValue(m_config.readNumEntry("FirstWait",30)*10);
   m_maxPingsAtOnce->setValue(m_config.readNumEntry("MaxPingsAtOnce",256));
   m_updatePeriod->setValue(m_config.readNumEntry("UpdatePeriod",300));
   m_pingAddresses->setText(m_config.readEntry("PingAddresses","192.168.0.0/255.255.255.0;192.168.100.0-192.168.100.254"));
   m_allowedAddresses->setText(m_config.readEntry("AllowedAddresses","192.168.0.0/255.255.0.0"));
   m_broadcastNetwork->setText(m_config.readEntry("BroadcastNetwork","192.168.0.0/255.255.0.0"));
   m_pingNames->clear();
   m_pingNames->insertStringList(m_config.readListEntry("PingNames",';'));
   int i=m_config.readNumEntry("SearchUsingNmblookup",1);
   m_useNmblookup->setChecked(i!=0);
};

void LisaSettings::save()
{
   if ( getuid()==0)
   {
      if (m_secondScan->isChecked())
         m_config.writeEntry("SecondWait",(m_secondWait->value()+5)/10);
      else
         m_config.writeEntry("SecondWait",-1);

      if (m_useNmblookup->isChecked())
         m_config.writeEntry("SearchUsingNmblookup",1);
      else
         m_config.writeEntry("SearchUsingNmblookup",0);

      if (m_deliverUnnamedHosts->isChecked())
         m_config.writeEntry("DeliverUnnamedHosts",1);
      else
         m_config.writeEntry("DeliverUnnamedHosts",0);

      m_config.writeEntry("FirstWait",(m_firstWait->value()+5)/10);
      m_config.writeEntry("MaxPingsAtOnce",m_maxPingsAtOnce->value());
      m_config.writeEntry("UpdatePeriod",m_updatePeriod->value());
      m_config.writeEntry("PingAddresses",m_pingAddresses->text());
      m_config.writeEntry("AllowedAddresses",m_allowedAddresses->text());
      m_config.writeEntry("BroadcastNetwork",m_broadcastNetwork->text());
      QStringList writeStuff;
      for (int i=0; i<m_pingNames->count(); i++)
         writeStuff.append(m_pingNames->text(i));
      m_config.writeEntry("PingNames",writeStuff,';');

      m_config.sync();
      chmod(QFile::encodeName(m_configFilename),S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
   }
   else
   {
      //ok, now it gets harder
      //we are not root but we want to write into /etc ....
      //any idea how to do it better ?
      QString username("???");
      struct passwd *user = getpwuid( getuid() );
      if ( user )
         username=user->pw_name;
      m_tmpFilename=QString("/tmp/kcmlisa-%1-%2-%3").arg(username).arg(getpid()).arg(time(0));

      QFile tmpFile(m_tmpFilename);
      bool res=tmpFile.open(IO_ReadWrite);
      if (res)
      {
         QTextStream confStream(&tmpFile);
         if (m_secondScan->isChecked())
            confStream<<"SecondWait = "<<(m_secondWait->value()+5)/10<<"\n";
         else
            confStream<<"SecondWait = -1\n";

         if (m_useNmblookup->isChecked())
            confStream<<"SearchUsingNmblookup = 1\n";
         else
            confStream<<"SearchUsingNmblookup = 0\n";

         if (m_deliverUnnamedHosts->isChecked())
            confStream<<"DeliverUnnamedHosts = 1\n";
         else
            confStream<<"DeliverUnnamedHosts = 0\n";

         confStream<<"FirstWait = "<< (m_firstWait->value()+5)/10 <<"\n";
         confStream<<"MaxPingsAtOnce = "<<m_maxPingsAtOnce->value()<<"\n";
         confStream<<"UpdatePeriod = "<<m_updatePeriod->value()<<"\n";
         confStream<<"PingAddresses = "<<m_pingAddresses->text().latin1()<<"\n";
         confStream<<"AllowedAddresses = "<<m_allowedAddresses->text().latin1()<<"\n";
         confStream<<"BroadcastNetwork = "<<m_broadcastNetwork->text().latin1()<<"\n";
         QString writeStuff;
         for (int i=0; i<m_pingNames->count(); i++)
            writeStuff=writeStuff+m_pingNames->text(i).latin1()+";";

         confStream<<"PingNames = "<<writeStuff.latin1()<<"\n";
         tmpFile.close();
         QString suCommand=QString("cp %1 %2; chmod 644 %3").arg(m_tmpFilename).arg(m_configFilename).arg(m_configFilename);
         KProcess *proc = new KProcess();
         connect(proc, SIGNAL(processExited(KProcess *)), this, SLOT(saveDone(KProcess *)));
         *proc<<"kdesu"<<"-c"<<suCommand;
         KApplication::setOverrideCursor(Qt::waitCursor);
         setEnabled(false);
         proc->start();
      }
      else
         KMessageBox::sorry(0,i18n("Sorry, saving the results to %1 failed.").arg(m_configFilename));
   }
}

void LisaSettings::autoSetup()
{
   LisaConfigInfo lci;
   SetupWizard *w=new SetupWizard(this,&lci);
   int result=w->exec();
   delete w;
   if (result!=QDialog::Accepted)
      return;

   m_pingAddresses->setText(lci.pingAddresses);
   m_broadcastNetwork->setText(lci.broadcastNetwork);
   m_allowedAddresses->setText(lci.allowedAddresses);
   m_secondWait->setValue(lci.secondWait*10);
   m_secondScan->setChecked(lci.secondScan);
   m_secondWait->setEnabled(lci.secondScan);
   m_firstWait->setValue(lci.firstWait*10);
   m_maxPingsAtOnce->setValue(lci.maxPingsAtOnce);
   m_updatePeriod->setValue(lci.updatePeriod);
   m_useNmblookup->setChecked(lci.useNmblookup);
   m_deliverUnnamedHosts->setChecked(lci.unnamedHosts);

   emit changed();
   return;
};

void LisaSettings::saveDone(KProcess *proc)
{
   unlink(QFile::encodeName(m_tmpFilename));
   KApplication::restoreOverrideCursor();
   setEnabled(true);
   delete(proc);
}

#include "kcmlisa.moc"

