// -*- c++ -*-

/*****************************************************************

Copyright (c) 1996-2002 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#ifndef KASTASKER_H
#define KASTASKER_H

#include "kasbar.h"

class KConfig;
class KPixmap;

class TaskManager;
class Task;
class Startup;

class KasTaskItem;
class KasStartupItem;
class KasGroupItem;

/**
 * A KasBar that provides a taskbar using the TaskManager API.
 *
 * @version $Id: kastasker.h,v 1.18 2002/10/27 13:53:10 mlaurent Exp $
 * @author Richard Moore, rich@kde.org
 */
class KasTasker : public KasBar
{
    Q_OBJECT

public:
    /**
     * Create a KasTasker widget.
     */
    KasTasker( Orientation o, QWidget* parent = 0, const char* name = 0, WFlags f = 0 );

    /**
     * Create a KasTasker widget that is slaved to another KasTasker. The
     * created widget will inherit the settings of the parent, but will
     * not connect to the signals of the TaskManager.
     */
    KasTasker( Orientation o, KasTasker *master,
	       QWidget *parent=0, const char *name=0, WFlags f=0 );
    /**
     * Cleans up.
     */
    virtual ~KasTasker();

    /**
     * Finds the item representing a task (if there is one).
     */
    KasTaskItem *findItem( Task * );

    /**
     * Finds the item representing a startup (if there is one).
     */
    KasStartupItem *findItem( Startup *s );

    /**
     * Accessor for the min icon (singleton).
     */
    QBitmap *minIcon();

    /**
     * Accessor for the max icon (singleton).
     */
    QBitmap *maxIcon();

    /**
     * Accessor for the shade icon (singleton).
     */
    QBitmap *shadeIcon();

    /**
     * Accessor for the micro min icon (singleton).
     */
    QPixmap *microMinIcon();

    /**
     * Accessor for the micro max icon (singleton).
     */
    QPixmap *microMaxIcon();

    /**
     * Accessor for the micro shade icon (singleton).
     */
    QPixmap *microShadeIcon();

    /**
     * Returns true iff thumbnails are enabled.
     */
    bool thumbnailsEnabled() const { return enableThumbs_; }

    /**
     * Returns the proportions of the window thumbnails.
     */
    double thumbnailSize() const { return thumbnailSize_; }

    /**
     * Returns true iff the startup notifier is enabled.
     */
    bool notifierEnabled() const { return enableNotifier_; }

    /**
     * Returns true iff the modified flag should be shown.
     */
    bool showModified() const { return showModified_; }

    /**
     * Returns true iff windows from all desktops should be displays.
     */
    bool showAllWindows() const { return showAllWindows_; }

    /**
     * Returns the delay between thumbnail updates (in seconds).
     */
    int thumbnailUpdateDelay() const { return thumbUpdateDelay_; }

    /**
     * Returns true iff windows should be grouped together.
     */
    bool groupWindows() const { return groupWindows_; }

    //
    // Internal stuff
    //

    /**
     * Returns the TaskManager object.
     */
    TaskManager *taskManager() const { return manager; }

    /**
     * Converts the item for a task into a group item to which additional
     * tasks can be added.
     */
    KasGroupItem *convertToGroup( Task *t );

    /**
     * Moves an item from a group into the main bar.
     */
    void moveToMain( KasGroupItem *gi, Task *t );

    /**
     * If the task should be part of a group then make it so.
     */
    KasItem *maybeAddToGroup( Task *t );

public slots:
    /**
     * Adds a task to the bar.
     */
    void addTask( Task * );

    /**
     * Removes a task from the bar.
     */
    void removeTask( Task * );

    /**
     * Adds a startup item to the bar.
     */
    void addStartup( Startup * );

    /**
     * Removes a startup item from the bar.
     */
    void removeStartup( Startup * );

    void refreshAll();
    void refreshIconGeometry();

    void setNotifierEnabled( bool enable );
    void setThumbnailSize( double size );
    void setThumbnailSize( int percent );
    void setThumbnailsEnabled( bool enable );
    void setShowModified( bool enable );
    void setShowAllWindows( bool enable );
    void setThumbnailUpdateDelay( int secs );
    void setGroupWindows( bool enable );

    void showPreferences();
    void showAbout();

    /**
     * Sets the current KConfig object.
     */
    void setConfig( KConfig *config );
    KConfig *config() const { return conf; }


    /**
     * Reads the settings from the current KConfig.
     */
    void readConfig();

    /**
     * Rereads the configuration of the master Kasbar.
     */
    void rereadMaster();

protected slots:
    /**
     * Load settings from the specified configuration.
     */
    void readConfig( KConfig *conf );

signals:
    /**
     * Emitted when the configuration has changed, and childs bars should
     * reread the master.
     */
    void configChanged();

private:
    KasTasker *master_;
    TaskManager *manager;
    QBitmap *minPix;
    QBitmap *maxPix;
    QBitmap *shadePix;
    QPixmap *microShadePix;
    QPixmap *microMaxPix;
    QPixmap *microMinPix;
    bool passive_;
    bool enableThumbs_;
    double thumbnailSize_;
    bool enableNotifier_;
    bool showModified_;
    bool showAllWindows_;
    int thumbUpdateDelay_;
    bool groupWindows_;
    KConfig *conf;
};

#endif // KASTASKER_H

