// -*- mode: c++; c-basic-offset: 4 -*-
/*
  Copyright (c) 2008 Laurent Montel <montel@kde.org>
  Copyright (C) 2006 Daniele Galdi <daniele.galdi@gmail.com>

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2 of the License, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 51
  Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "adblock.h"
#include "adblockdialog.h"

#include <kdebug.h>
#include <kmenu.h>
#include <klocale.h>
#include <KTreeWidgetSearchLine>
#include <KRun>

#include <qcursor.h>
#include <qlabel.h>
#include <KVBox>
#include <qlineedit.h>
#include <qcolor.h>
#include <qfont.h>
#include <qpainter.h>
#include <QTreeWidget>
#include <QApplication>
#include <QClipboard>

AdBlockDlg::AdBlockDlg(QWidget *parent, AdElementList &elements, KHTMLPart*part) :
    KDialog( parent ), m_part( part )
{
    setModal( true );
    setCaption( i18n("Adblock dialog") );
    setButtons( KDialog::Ok | KDialog::Cancel );
    setDefaultButton(KDialog::Ok);



    KVBox *page = new KVBox(this);
    setMainWidget(page);
    m_label1 = new QLabel( i18n("All blockable items in this page:"), page);

    KTreeWidgetSearchLine* searchLine = new KTreeWidgetSearchLine( page );

    m_list = new QTreeWidget(page);
    m_list->setAllColumnsShowFocus( true );

    searchLine->setTreeWidget( m_list );

    QStringList lstHeader;
    lstHeader<<i18n("Source")<<i18n("Category")<<i18n("Node Name");
    m_list->setHeaderLabels(lstHeader);

    m_list->setColumnWidth(0, 600);
    m_list->setColumnWidth(1, 90);
    m_list->setColumnWidth(2, 90);

    m_list->setRootIsDecorated( false );
    AdElementList::iterator it;
    for ( it = elements.begin(); it != elements.end(); ++it )
    {
	AdElement &element = (*it);

        QStringList lst;
        lst<<element.url()<<  element.category()<<element.type();

        ListViewItem *item = new ListViewItem( m_list,lst );
	item->setBlocked(element.isBlocked());
        item->setNode( element.node() );
    }

    m_label2 = new QLabel( i18n("New filter (use * as a wildcard):"), page);

    m_filter = new QLineEdit( page);

    connect(this, SIGNAL( okClicked() ), this, SLOT( validateFilter() ));
    connect(m_list, SIGNAL( itemDoubleClicked(QTreeWidgetItem *, int )), this, SLOT(updateFilter(QTreeWidgetItem *)) );

    m_menu = new KMenu(this);
    m_menu->addAction(i18n("Filter this item"), this, SLOT(filterItem()));
    m_menu->addAction(i18n("Filter all items at same path"), this, SLOT(filterPath()));
    m_menu->addAction( i18n( "Add this item to white list" ), this, SLOT( addWhiteList() ) );
    m_menu->addSeparator();
    m_menu->addAction( i18n( "Copy Link Address" ),  this,  SLOT( copyLinkAddress() ) );
    //comment for the moment
    //m_menu->addAction( i18n( "Highlight Element" ), this, SLOT( highLightElement() ) );
    m_menu->addAction( i18n( "Show it" ), this, SLOT( showElement() ) );
    m_list->setContextMenuPolicy( Qt::CustomContextMenu );
    connect( m_list, SIGNAL( customContextMenuRequested( const QPoint & ) ),
           this, SLOT( showContextMenu(const QPoint & ) ) );

    resize( 800, 400 );
}

void AdBlockDlg::updateFilter(QTreeWidgetItem *selected)
{
    ListViewItem *item = static_cast<ListViewItem *>(selected);

    if (item->isBlocked())
    {
	m_filter->clear();
	return;
    }

    m_filter->setText( item->text(0) );
}

void AdBlockDlg::validateFilter()
{
    const QString text = m_filter->text().trimmed();

    if (!text.isEmpty())
        emit notEmptyFilter(text);

    delayedDestruct();
}

void AdBlockDlg::showContextMenu(const QPoint & pos)
{
    QPoint newPos = m_list->viewport()->mapToGlobal( pos );
    int column = m_list->columnAt(pos.x() );
    if (column == -1)
        return;
    m_menu->popup(newPos);
}

void AdBlockDlg::filterItem()
{
    QTreeWidgetItem* item = m_list->currentItem();
    m_filter->setText( item->text(0) );
}

void AdBlockDlg::filterPath()
{
    QTreeWidgetItem* item = m_list->currentItem();
    QString value = item->text(0);
    m_filter->setText( value.section( '/', 0, -2 ).append("/*") );
}

void AdBlockDlg::addWhiteList()
{
    QTreeWidgetItem* item = m_list->currentItem();
    m_filter->setText( "@@" + item->text(0) );
}

void AdBlockDlg::copyLinkAddress()
{
    QApplication::clipboard()->setText(m_list->currentItem()->text( 0 )  );
}

void AdBlockDlg::highLightElement()
{
    ListViewItem *item = static_cast<ListViewItem *>(m_list->currentItem());
    if ( item )
    {
        DOM::Node handle = item->node();
        kDebug()<<" m_part :"<<m_part;
        if (!handle.isNull()) {
            m_part->setActiveNode(handle);
        }
    }
}

void AdBlockDlg::showElement()
{
    new KRun( m_list->currentItem()->text( 0 ), 0 );
}

AdBlockDlg::~AdBlockDlg()
{
}

// ----------------------------------------------------------------------------
bool ListViewItem::isBlocked() const
{
    return m_blocked;
}

void ListViewItem::setBlocked(bool blocked)
{
    if ( blocked )
    {
        setData ( 0,Qt::TextColorRole, Qt::red);
        QFont itemFont =  font(0);
        itemFont.setItalic( true );
        itemFont.setBold( true );
        setData( 0, Qt::FontRole, itemFont );

    }
}

DOM::Node ListViewItem::node() const
{
    return m_node;
}

void ListViewItem::setNode( const DOM::Node& node )
{
    m_node = node;
}

#include "adblockdialog.moc"
